﻿using System;
using UnigeFrontEnd.Tools;
using Newtonsoft.Json;
using System.IO;
using System.Collections.Generic;

namespace BabyMind.DAQ
{

	/// <summary>
	/// JSON Mini-crate settings 
	/// </summary>
	public class AppDataSettings
	{
		public class MCRSettings
		{
			/// <summary>
			/// JSON Board ID association settings
			/// </summary>
			public class UsbBoardIdSettings
			{
				public byte BoardIdStart = 0;
				public byte BoardIdEnd = 0;
			}
			/// <summary>
			/// JSON Configuration settings
			/// </summary>
			public class ConfigurationSettings
			{
				public bool DefaultConfigExtXML = false;
				public byte DefaultSelDevices = 255;
				public bool EnableThread = true;
			}

			/// <summary>
			/// JSON Readout settings
			/// </summary>
			public class ReadoutSettings
			{
				public bool MonitoringQueue = false;
				public bool UDPReadout = false;
				public UInt16 UDPport = 11000;
				public byte[] UDP_IPaddress = { 0, 0, 0, 0 };
			}

			public UsbBoardIdSettings UsbBoardId = new UsbBoardIdSettings();
			public ConfigurationSettings Configuration = new ConfigurationSettings();
			public ReadoutSettings Readout = new ReadoutSettings();
			public List<string> ConfigurationFiles = new List<string>();
			public string DaqPath = "D:/Data/";
			public string DaqType = "cosmics/";
			public string DaqFileName = "test_MCR_";
			public string RunNumber = "0";
			public string FullName = "test.daq";
			public int FileSizeLimit = 1000000; //in KB
			public int TimeLimit = 10000; //in ms
		}

		// JSON environment settings
		public class EnvironmentSettings
		{
			/// <summary>
			/// JSON Configuration settings
			/// </summary>
			public class ConfigurationSettings
			{
				public byte[] IPaddress = { 0, 0, 0, 0 };
				public UInt16 IPport = 11000;
				public UInt16 ForceDisconnectIPport = 12000;
				public string ForceDisconnectPassword = "HighwayToHell";
				public bool DebugDisplay = true;
				public string MCBComPort = "COM1";
                public Lib.MCBLib.MCBDAQmodes MCB_mode = Lib.MCBLib.MCBDAQmodes.EXT_IN_NIM1;
                public bool InternalSpill = true;
				public byte InternalDAQPulseWidth = 1;
				public byte BeamDAQPulseWidth = 0;
            }
			/// <summary>
			/// JSON Error Thread settings
			/// </summary>
			public class ErrorThreadSettings
			{
				public UInt16 IPport = 11001;
				public int QueueSize = 2000;
				public Logger.Severity SeverityLevel = UnigeFrontEnd.Tools.Logger.Severity.CRITICAL;
				public bool Enabled = true;
			}

			/// <summary>
			/// JSON Monitoring Thread settings
			/// </summary>
			public class MonitoringThreadSettings
			{
				public bool ReducedMonitoring = false;
				public UInt16 IPport = 11002;
				public int PeriodSeconds = 10;
			}
			/// <summary>
			/// JSON Logger settings
			/// </summary>
			public class LoggerSettings
			{

				public string AppPath = "$APP$/LOG/APP";
				public string CmdPath = "$APP$/LOG/CMD";
				public string ErrorPath = "$APP$/LOG/APP";
				public string AppDAQPrefix = "DAQ";
				public string ErrorPrefix = "Error";
				public string TCPIPCmdPrefix = "TCPIP";
				public string ThreadCmdPrefix = "Thread";
			}
			public ConfigurationSettings Configuration =  new ConfigurationSettings();
			public LoggerSettings Logger = new LoggerSettings();
			public ErrorThreadSettings ErrorThread = new ErrorThreadSettings();
			public MonitoringThreadSettings MonitoringThread = new MonitoringThreadSettings();

			/// <summary>
			/// Will check or create a directory and create a log file of type x_path/x_filePrefix_DateTime.log
			/// </summary>
			/// <param name="x_path">path to check or create</param>
			/// <param name="x_filePrefix">file prefix</param>
			/// <returns>full file name + directory</returns>
			public static string CreateDirAndFile(string x_path, string x_filePrefix)
			{
				string l_logPath = x_path;
				if (x_path.StartsWith("$APP$"))
					l_logPath = x_path.Replace("$APP$", Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location));

				return UnigeFrontEnd.Tools.Logger.CreateDirAndFile(l_logPath, x_filePrefix);
			}
		}
	}
	/// <summary>
	/// Application setting from JSON setting file
	/// </summary>
	public class AppSettings
	{
		/// <summary>
		/// The JSON description to be de-serialized
		/// </summary>
		public class Data
		{
			public AppDataSettings.EnvironmentSettings Environment = new AppDataSettings.EnvironmentSettings();
			public List<AppDataSettings.MCRSettings> MCR = new List<AppDataSettings.MCRSettings>();

			[JsonIgnore()]
			public string JsonFilename { internal set; get; }

			/// <summary>
			/// Display AppSettings variables
			/// </summary>
			/// <returns>string to be displayed</returns>
			public string DisplayAllVariables()
			{
				string l_str = "Environment:\n";
				l_str += $" IPaddress: {Environment.Configuration.IPaddress[0]}.{Environment.Configuration.IPaddress[1]}.{Environment.Configuration.IPaddress[2]}.{Environment.Configuration.IPaddress[3]}\n";
				l_str += " Configuration.IPport:" + Environment.Configuration.IPport + "\n";
				l_str += " Configuration.MCBComPort:" + Environment.Configuration.MCBComPort + "\n";
				l_str += " ErrorThreadSettings.IPport:" + Environment.ErrorThread.IPport + "\n";
				l_str += " ErrorThreadSettings.QueueSize:" + Environment.ErrorThread.QueueSize + "\n";
				l_str += " ErrorThreadSettings.SeverityLevel:" + Environment.ErrorThread.SeverityLevel + "\n";
				l_str += " ErrorThreadSettings.Enabled:" + Environment.ErrorThread.Enabled + "\n";
				l_str += " MonitoringThread.IPport:" + Environment.MonitoringThread.IPport + "\n";
				l_str += " MonitoringThread.PeriodSeconds:" + Environment.MonitoringThread.PeriodSeconds + "\n";
				l_str += " MonitoringThread.ReducedMonitoring:" + Environment.MonitoringThread.ReducedMonitoring + "\n";
				l_str += " Logger.AppPath:" + Environment.Logger.AppPath + "\n";
				l_str += " Logger.CmdPath:" + Environment.Logger.CmdPath + "\n";
				l_str += " Logger.ErrorPath:" + Environment.Logger.ErrorPath + "\n";
				l_str += "\n";
				l_str += "MCR list:\n";
				foreach (var l_mcr in MCR)
				{
					l_str += "  RelativeBoardIdStart:" + l_mcr.UsbBoardId.BoardIdStart + "\n";
					l_str += "  RelativeBoardIdEnd:" + l_mcr.UsbBoardId.BoardIdEnd + "\n";
					l_str += "  Readout.UDPReadout:" + l_mcr.Readout.UDPReadout + "\n";
					l_str += "  Readout.UDPPort:" + l_mcr.Readout.UDPport + "\n";
					l_str += "  Configuration.EnableThread:" + l_mcr.Configuration.EnableThread + "\n";
					l_str += "  Configuration.DefaultSelDevices:" + l_mcr.Configuration.DefaultSelDevices + "\n";
					l_str += "\n";
				}
				return l_str;
			}
		}

		readonly Data m_data;
		public Data AppSettingsData { get { return m_data; } }

		/// <summary>
		/// Ctor
		/// </summary>
		/// <param name="x_configFilename">config file name to be used for JSON deserializing</param>
		/// <param name="x_configDefaultFilename">default config file name to be used for JSON deserializing if x_configFilename is not found</param>
		public AppSettings(string x_configFilename, string x_configDefaultFilename = null)
		{
			// Check filename function
			Func<string, string, string> l_buildAndCheckFilename = (x_filename, x_varName) =>
			{
				if (x_filename == null)
					throw new NullReferenceException($"The var name '{nameof(x_varName)}' in Settings function l_buildAndCheckFilename is null");

				if (string.IsNullOrWhiteSpace(x_filename))
					throw new ArgumentException(x_varName, "Empty or whitespace string for config filename");


				string x_pathname = Path.Combine(Utils.ApplicationPath, x_filename);

				FileInfo fi = null;
				try
				{
					fi = new FileInfo(x_pathname);
				}
				catch (ArgumentException) { }
				catch (PathTooLongException) { }
				catch (NotSupportedException) { }
				if (fi == null)
				{
					throw new ArgumentException(nameof(x_filename), "Invalid filename");
				}
				return x_pathname;
			};

			// Get full pathnames
			string l_filename = l_buildAndCheckFilename(x_configFilename, nameof(x_configFilename));
			string l_defaultFilename = (x_configDefaultFilename == null ? null : l_buildAndCheckFilename(x_configDefaultFilename, nameof(x_configDefaultFilename)));

			// Load settings if it exists
			if (File.Exists(l_filename))
			{
				m_data = JsonConvert.DeserializeObject<Data>(File.ReadAllText(l_filename));
			}
			// Or load default settings if it exists
			else if ((l_defaultFilename != null) && File.Exists(l_defaultFilename))
			{
				m_data = JsonConvert.DeserializeObject<Data>(File.ReadAllText(l_defaultFilename));
			}
			// Or load defaults
			else
			{
				m_data = new Data();
				Save();
			}
			m_data.JsonFilename = l_filename;
		}

		/// <summary>
		/// Automatically called after any change on the lib settings
		/// To be call by application if some settings must be saved after a user action
		/// </summary>
		public void Save()
		{
			string l_json = JsonConvert.SerializeObject(m_data, Formatting.Indented);
			File.WriteAllText(m_data.JsonFilename, l_json);
		}
	}
}
