﻿using System;
using System.Collections.Generic;
using BabyMind.Lib;
using System.Threading;
using UnigeFrontEnd.Tools;

namespace BabyMind.DAQ
{
	/// <summary>
	/// class used to pass BabyMind lib objects
	/// </summary>
	public class BabyMindLib
	{
		/// <summary>
		/// Lib board Link
		/// </summary>
		public LibBoardLink Link { get; }
		/// <summary>
		/// Chain/TDM/FBW macro functions 
		/// </summary>
		public ChainTdmFbwLib ChainTdmFbw { get; }
		/// <summary>
		/// Application settings for this MCR
		/// </summary>
		public AppDataSettings.MCRSettings MCRSettings { get; }

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_link">Lib board Link</param>
		/// <param name="x_ctf">Chain/TDM/FBW macro functions</param>
		/// <param name="x_settings">Application settings for this MCR</param>
		public BabyMindLib(LibBoardLink x_link, ChainTdmFbwLib x_ctf, AppDataSettings.MCRSettings x_settings)
		{
			Link = x_link;
			ChainTdmFbw = x_ctf;
			MCRSettings = x_settings;
		}
	}

	/// <summary>
	/// Command base class
	/// </summary>
	public abstract class GenericCommand
	{
		protected CommandEnv m_command;
		public static readonly string _ERR_RETURN = "[Error]:";

		/// <summary>
		/// Print the exception and recursive
		/// </summary>
		/// <param name="x_e">exception</param>
		/// <returns>string message returned from exception</returns>
		protected string printException(Exception x_e)
		{
			Exception l_inn = x_e;
			string l_msg = x_e.Message;
			while (l_inn.InnerException != null)
			{
				l_inn = l_inn.InnerException;
			}
			if (l_inn != x_e)
				l_msg += ": " + l_inn.Message + "\n";
			return l_msg;
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_name">Name of the command</param>
		/// <param name="x_command">Command object used to associate with</param>
		protected GenericCommand(CommandEnv x_command, string x_name)
		{
			Name = x_name;
			m_command = x_command;
			x_command.Add(this);
		}

		public bool DisableLog = false;
		public string Name { get; private set; }
		public abstract string Execute(string[] x_args);
		public abstract string help();
	}

	/// <summary>
	/// Thread command base class
	/// </summary>
	public abstract class ThreadCommand:GenericCommand
	{
		protected new ThreadCommandEnv m_command;
		
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_name">Name of the command</param>
		/// <param name="x_command">Command object used to associate with</param>
		protected ThreadCommand(ThreadCommandEnv x_command, string x_name):base(x_command, x_name)
		{
			m_command = x_command;
		}

	}

	/// <summary>
	/// TCP/IP command base class
	/// </summary>
	public abstract class TCPIPCommand : GenericCommand
	{
		protected new TCPIPCommandEnv m_command;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_name">Name of the command</param>
		/// <param name="x_command">Command object used to associate with</param>
		protected TCPIPCommand(TCPIPCommandEnv x_command, string x_name) : base(x_command, x_name)
		{
			m_command = x_command;
		}

	}

	/// <summary>
	/// Command Environment base class
	/// </summary>
	public class CommandEnv
	{
		protected List<GenericCommand> m_list;

		/// <summary>
		/// Logger associated to this thread
		/// </summary>
		public Logger Log { protected set; get; }

		/// <summary>
		/// Constructor
		/// </summary>
		public CommandEnv()
		{
			m_list = new List<GenericCommand>();
		}

		/// <summary>
		/// Add the command to the list
		/// </summary>
		/// <param name="x_cmd">command to be added</param>
		internal void Add(GenericCommand x_cmd)
		{
			m_list.Add(x_cmd);
		}

		/// <summary>
		/// Run the command from its name, including concurrent access from multiple thread protection
		/// </summary>
		///<param name="x_args">argument of the command line, x_args[0] = name</param>
		///<returns>string message returned in case of error</returns>
		public string RunCommand(string[] x_args)
		{
			lock(this)
			{
				string l_str = "";
				GenericCommand l_cmd = m_list.Find(x_c => (x_c.Name == x_args[0]));
				if (l_cmd != null)
				{
					if (!l_cmd.DisableLog)
						Log?.WriteLine(string.Join(" ", x_args), Logger.Severity.INFO);
					l_str = l_cmd.Execute(x_args);
				}
				else
				{
					l_str = $"Command '{x_args[0]}' not found\n";
					//l_str += Help();
				}
				if(!l_cmd.DisableLog)
					Log?.WriteLine(l_str, Logger.Severity.INFO);
				return l_str;
			}
		}

		/// <summary>
		/// help on all commands
		/// </summary>
		public string Help()
		{
			string l_str = "";
			foreach (var l_cmd in m_list)
			{
				l_str += l_cmd.Name + ":\n";
				l_str += l_cmd.help() + "\n\n";
			}
			return l_str;
		}
	}

	/// <summary>
	/// Command Environment associated to TCP/IP commands
	/// </summary>
	public class TCPIPCommandEnv : CommandEnv
	{
		/// <summary>
		/// List of MCR threads
		/// </summary>
		public IReadOnlyList<MCRThread> MCRList { get; }
		/// <summary>
		/// Serial Port
		/// </summary>
		public System.IO.Ports.SerialPort MCBSerialPort { get; }

        /// <summary>
        /// MCB mode
        /// </summary>
        public Lib.MCBLib.MCBDAQmodes MCBMode { get; }

        /// <summary>
        /// Enable internal spill
        /// </summary>
        public bool EnableInternalSpill { get; }
		/// <summary>
		/// MonitoringData for all MCRs and Boards
		/// </summary>
		public MonitoringData.Data MonitoringData { get; }

		/// <summary>
		/// Monitoring Thread
		/// </summary>
		public MonitoringThread MonitoringThread { get; }

		/// <summary>
		/// Monitoring ReducedData for Daq file size
		/// </summary>
		public MonitoringData.ReducedData MonitoringReducedData { get; }

		public bool ReducedMonitoringEnabled { get; }



		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_list">list of MCR thread</param>
		/// <param name="x_port">Serial port to be used with MCB</param>
		/// <param name="x_mcbMode">MCB mode</param>
		/// <param name="x_cmdLoggerPath">Commands Logger Path</param>
		/// <param name="x_filenamePrefixLogger">Prefix used for the logger filename</param>
		/// <param name="x_monitoringData">Monitoring data for all MCRs and Boards</param>
		/// <param name="x_monitoringReducedData">Reduced monitoring enabled</param>
		public TCPIPCommandEnv(IReadOnlyList<MCRThread> x_list, System.IO.Ports.SerialPort x_port,Lib.MCBLib.MCBDAQmodes x_mcbMode , bool x_enableInternalSpill ,string x_cmdLoggerPath, string x_filenamePrefixLogger, 
						MonitoringData.Data x_monitoringData,MonitoringData.ReducedData x_monitoringReducedData , MonitoringThread x_monitoringThread ,bool x_reducedMonitoringEnabled) : base()
		{
			MCRList = x_list;
			MCBSerialPort = x_port;
			MCBMode = x_mcbMode;
            EnableInternalSpill = x_enableInternalSpill;
			MonitoringData = x_monitoringData;
			MonitoringReducedData = x_monitoringReducedData;
			MonitoringThread = x_monitoringThread;
			ReducedMonitoringEnabled = x_reducedMonitoringEnabled;


			string l_logfile = AppDataSettings.EnvironmentSettings.CreateDirAndFile(x_cmdLoggerPath, x_filenamePrefixLogger);			
			Log = new Logger(l_logfile);
		}
	}

	/// <summary>
	/// Command environment associated to a thread execution
	/// </summary>
	public class ThreadCommandEnv:CommandEnv
	{
		/// <summary>
		/// BabyMind Library to access to hardware functions on this thread environment
		/// </summary>
		public BabyMindLib BMLib { get; }

		/// <summary>
		/// Logger with queue
		/// </summary>
		public new LoggerWithQueue Log { protected set; get; }

		/// <summary>
		/// Monitoring data for MCR
		/// </summary>
		public MonitoringData.Data.MCR MCR_data { get; }

		/// <summary>
		/// Monitoring reduced data for MCR
		/// </summary>
		public MonitoringData.ReducedData.MCR MCR_reducedData { get; }

		/// <summary>
		/// Get the return string, normally used only in asynchronous mode
		/// </summary>
		public string ReturnString { get; private set; }

		// used in sharing mode for synchronization
		static CountdownEvent m_endCountDownEv = new CountdownEvent(0);

		bool ExitFlag { get; set; }
		AutoResetEvent m_startEv, m_endEv;
		bool m_asynchronous = false;
		
		string[] m_args;


		public static readonly string _CMD_EXIT = "Exit";

		/// <summary>
		/// Constructor
		/// </summary>
		///<param name="x_lib">BabyMind lib objects</param> 
		///<param name="x_ThreadLog">Thread logger</param>
		///<param name="x_mcr">Monitoring data MCR</param>
		///<param name="x_mcrReducedData">Monitoring reduced data MCR</param>
		public ThreadCommandEnv(BabyMindLib x_lib, LoggerWithQueue x_ThreadLog, MonitoringData.Data.MCR x_mcr, MonitoringData.ReducedData.MCR x_mcrReducedData) : base()
		{
			BMLib = x_lib;
			MCR_data = x_mcr;
			MCR_reducedData = x_mcrReducedData;
			Log = x_ThreadLog;
			ExitFlag = false;
			m_startEv = new AutoResetEvent(false);
			m_endEv = new AutoResetEvent(false);
		}

		/// <summary>
		/// Configure the next commands to be used in asynchronous mode and set number of asynchronous thread commands to be signaled
		/// RunAsynchronousCommand and WaitAsynchronous must be used
		/// </summary>
		/// <param name="x_count">Initial count property used by WaitAsynchronous</param>
		public static void ResetAsynchronous(int x_count)
		{
			m_endCountDownEv.Reset(x_count);
		}

		/// <summary>
		/// Wait for all flags set in ResetAsynchronous to be signaled
		/// The string returned by each thread are put into the string array passed in ResetAsynchronous
		/// </summary>
		public static void WaitAsynchronous()
		{
			m_endCountDownEv.Wait();
		}

		/// <summary>
		/// Run asynchronously the command from its name within the non thread environment,
		/// including concurrent access from multiple thread protection. 
		/// ResetAsynchronous must be call first
		/// </summary>
		///<param name="x_args">argument of the command line, x_args[0] = name</param>
		///<returns>string message returned in case of error</returns>
		public string RunAsynchronousCommand(string[] x_args)
		{
			lock (this)
			{
				string l_str = "";

				if((m_startEv == null) || (m_endCountDownEv == null))
					return "Start and/or End Reset Event(s) from Thread are null\n";

				if(m_endCountDownEv.CurrentCount == 0)
					return "ResetAsynchronous has not initialized x_count parameter\n";

				m_asynchronous = true;
				// pass arguments to thread
				m_args = x_args;
				// launch start signal to thread
				m_startEv.Set();
				
				return l_str;
			}
		}


		/// <summary>
		/// Run the command from its name within the non thread environment,
		/// including concurrent access from multiple thread protection
		/// </summary>
		///<param name="x_args">argument of the command line, x_args[0] = name</param>
		///<returns>string message returned in case of error</returns>
		public new string RunCommand(string[] x_args)
		{
			lock (this)
			{
				if (m_asynchronous)
					m_endCountDownEv.Wait();

				string l_str = "";
				m_asynchronous = false;
				if ((m_startEv != null) && (m_endEv != null))
				{
					// pass arguments to thread
					m_args = x_args;
					// launch start signal to thread
					m_startEv.Set();					
					// wait for end of thread command and auto-reset it
					m_endEv.WaitOne();
					// get returned string
					l_str = ReturnString;
				}
				else
				{
					l_str = "Start and/or End Reset Event(s) from Thread are null\n";
				}
				return l_str;
			}
		}

		/// <summary>
		/// Help
		/// </summary>
		public class Exit : ThreadCommand
		{
			public Exit(ThreadCommandEnv x_command) : base(x_command, _CMD_EXIT) { }

			public override string Execute(string[] x_args)
			{
				m_command.ExitFlag = true;
				return "Exiting the thread " + Thread.CurrentThread.Name +"\n";
			}
			public override string help()
			{
				return "Exit the thread";
			}
		}

		/// <summary>
		/// Run the command interpreter in the thread environment, while loop included
		/// Assume all x_thread & x_lib objects created before entering else throw exception
		/// </summary>		
		public void RunThreadCommands()
		{
			if ((m_startEv == null) || (m_endEv == null) || (m_endCountDownEv == null))
				throw new NullReferenceException("Start and/or End Reset Event(s) from Thread are null");

			if ((BMLib.Link == null) || (BMLib.ChainTdmFbw == null))
				throw new NullReferenceException("LibBoardLink and/or ChainTdmFbwLib from Thread are null");

			// entering thread loop
			while (!ExitFlag)
			{
				// wait for start signal from thread caller and auto-reset it
				m_startEv.WaitOne();
				GenericCommand l_cmd = m_list.Find(x_c => (x_c.Name == m_args[0]));
				if (l_cmd != null)
				{
					try
					{
						if(!l_cmd.DisableLog)
							Log?.WriteLine(string.Join(" ", m_args), Logger.Severity.INFO);
						ReturnString = l_cmd.Execute(m_args);
						if (!l_cmd.DisableLog)
							Log?.WriteLine(ReturnString, Logger.Severity.INFO);
					}
					catch (Exception x_e)
					{
						ReturnString = GenericCommand._ERR_RETURN + $"[Unhandled exception : {x_e.Message}]";
						Log?.WriteLine(ReturnString, Logger.Severity.FAILURE);
					}
				}
				else
				{
					ReturnString = GenericCommand._ERR_RETURN + $"[Command '{m_args[0]}' not found]";
					Log?.WriteLine(ReturnString, Logger.Severity.CRITICAL);
				}

				// launch end signal to thread caller
				if (m_asynchronous)
					m_endCountDownEv.Signal();
				else
					m_endEv.Set();

				m_asynchronous = false;
			}
		}
	}
}
