﻿using System;
using System.Text;
using System.Threading;
using UnigeFrontEnd.Tools;
using System.Net.Sockets;

namespace BabyMind.DAQ
{
	/// <summary>
	/// Socket extension class
	/// </summary>
	public static class SocketExtensions
	{
		/// <summary>
		/// Return is socket is connected
		/// </summary>
		/// <param name="socket"></param>
		/// <returns>true if connected</returns>
		public static bool IsConnected(this Socket socket)
		{
			try
			{
				return !(socket.Poll(1, SelectMode.SelectRead) && socket.Available == 0);
			}
			catch (SocketException) { return false; }
		}
	}

	class LogErrorThread
	{
		/// <summary>
		/// Get the list of the concurrent Queue threads
		/// </summary>
		public System.Collections.Concurrent.ConcurrentQueue<string>[] QueueArray { get; private set; }

        System.Collections.Concurrent.ConcurrentQueue<string>[] m_allQueueArray;

        AppDataSettings.EnvironmentSettings.ErrorThreadSettings m_setting;
		Logger m_log;
		Logger m_appLog;

		// socket objects
		System.Net.IPAddress m_ip;
		Socket m_listener;
		Socket m_connectedSocket;
		bool m_stop;

		Thread m_thread;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_settings">LogError Settings from App Settings file</param>
		/// <param name="x_appLog">Main application Logger</param>
        /// <param name="x_appQueue">Main app log queue</param>
		/// <param name="x_nbQueue">nb of queue to create and try to dequeue</param>
		/// <param name="x_cmdLoggerPath">Commands Logger Path</param>
		/// <param name="x_filenamePrefixLogger">Prefix used for the logger filename</param> 
		/// <param name="x_ip">IP address</param>
		public LogErrorThread(AppDataSettings.EnvironmentSettings.ErrorThreadSettings x_settings, LoggerWithQueue x_appLog,System.Collections.Concurrent.ConcurrentQueue<string> x_appQueue, int x_nbQueue, string x_cmdLoggerPath, string x_filenamePrefixLogger, System.Net.IPAddress x_ip)
		{
			m_setting = x_settings;
			m_appLog = x_appLog;
			m_ip = x_ip;

			string l_logfile = AppDataSettings.EnvironmentSettings.CreateDirAndFile(x_cmdLoggerPath, $"{x_filenamePrefixLogger}_");
			m_appLog.WriteLine($"Building LogError Thread", Logger.Severity.INFO);

			m_log = new Logger(l_logfile);

			QueueArray = new System.Collections.Concurrent.ConcurrentQueue<string>[x_nbQueue];
            m_allQueueArray = new System.Collections.Concurrent.ConcurrentQueue<string>[x_nbQueue + 1];

            if (QueueArray == null)
			{
				m_appLog.WriteLine($"Impossible to allocate array of {x_nbQueue} error queues", Logger.Severity.FAILURE);
				return;
			}

			for (int l_i = 0; l_i < x_nbQueue; l_i++)
			{
				QueueArray[l_i] = new System.Collections.Concurrent.ConcurrentQueue<string>();
				if (QueueArray[l_i] == null)
				{
					m_appLog.WriteLine($"Impossible to allocate error queue #{l_i}", Logger.Severity.FAILURE);
					return;
				}
                m_allQueueArray[l_i] = QueueArray[l_i];
            }
            m_allQueueArray[x_nbQueue] = x_appQueue;

            m_appLog.WriteLine($"Allocated {m_allQueueArray.Length} error queues", Logger.Severity.INFO);

			if (x_settings.Enabled)
			{
				// create the error thread for try to dequeue data and push it to TCP/IP
				m_thread = new Thread(new ThreadStart(() => RunThread()));
				m_thread.Priority = ThreadPriority.Normal;
				m_thread.Name = "ErrorThread";
				m_thread.Start();
			}else
				m_appLog.WriteLine($"LogError Thread disabled", Logger.Severity.INFO);
		}

		/// <summary>
		/// Stop the socket server
		/// </summary>
		public void Stop()
		{
			m_log.WriteLine($"LogError Thread stop request", Logger.Severity.INFO);
			m_stop = true;
			m_thread.Abort();
			m_thread.Join();
			m_connectedSocket?.Close();
			m_listener?.Close();
		}

		/// <summary>
		/// Main Core of the thread
		/// </summary>
		void RunThread()
		{
			m_appLog.WriteLine($"LogError Thread started", Logger.Severity.INFO);

			System.Net.IPHostEntry l_ipHostInfo = System.Net.Dns.GetHostEntry(System.Net.Dns.GetHostName());

			System.Net.IPEndPoint l_localEP = new System.Net.IPEndPoint(m_ip, m_setting.IPport);
			m_appLog.WriteLine($"LogError Thread Server running on IP Address {l_localEP.Address} / Port {l_localEP.Port}", Logger.Severity.INFO);
			m_log.WriteLine($"Server running on IP Address {l_localEP.Address} / Port {l_localEP.Port}", Logger.Severity.INFO);

			// create the listener socket & bind it to IP/Port
			m_listener = new Socket(l_localEP.Address.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
			m_listener.Bind(l_localEP);
			m_stop = false;
			m_listener.Listen(1); // 1 client allowed per application


			while (!m_stop)
			{
				try
				{
					m_log.WriteLine("Server waiting for a new single client connection", Logger.Severity.INFO);
					m_connectedSocket = m_listener.Accept();
					m_appLog.WriteLine("LogError Thread Server connected to remote client", Logger.Severity.INFO);
					m_log.WriteLine("Server connected to remote client", Logger.Severity.INFO);

					int l_queue = 0;
					while (SocketExtensions.IsConnected(m_connectedSocket) && (!m_stop))
					{
						string l_str;
						// try to dequeue the current Queue and send it to socket
						while (m_allQueueArray[l_queue].TryDequeue(out l_str))
						{
                            //write in Error Log
                            m_log.WriteLine(l_str.Insert(l_str.IndexOf("]:[") + 3, $"Thread #{l_queue}, "), Logger.Severity.INFO);

                            //insert thread info, convert to byte[] and remove LF
                            byte[] l_bytesToSend = Encoding.UTF8.GetBytes(l_str.Insert(l_str.IndexOf("]:[") + 3, $"Thread #{l_queue}, "));
							if (SocketExtensions.IsConnected(m_connectedSocket) && (!m_stop))
								m_connectedSocket.Send(l_bytesToSend, 0, l_bytesToSend.Length, SocketFlags.None);
							else
								break;
						}
						// go to next queue
						l_queue++;
						if (l_queue >= m_allQueueArray.Length)
							l_queue = 0;

						Thread.Sleep(10);
						Thread.Yield();
					}
					// shutdown the socket
					//m_connectedSocket.Shutdown(SocketShutdown.Both);
					m_connectedSocket.Close();
					m_appLog.WriteLine("LogError Thread Server disconnected from remote client (from client stop)", Logger.Severity.CRITICAL);
					m_log.WriteLine("Server disconnected from remote client (from client stop)", Logger.Severity.CRITICAL);
				}
				catch (Exception)
				{
					m_connectedSocket.Close();
					m_appLog.WriteLine("LogError Thread Server disconnected from remote client (from exception)", Logger.Severity.CRITICAL);
					m_log.WriteLine("Server disconnected from remote client (from exception)", Logger.Severity.CRITICAL);
				}
			}

			m_appLog.WriteLine("LogError Thread Server is stopped", Logger.Severity.CRITICAL);
			m_log.WriteLine("Server is stopped", Logger.Severity.CRITICAL);
		}
	}
}
