﻿using System;
using System.Collections.Generic;
using BabyMind.Lib;
using UnigeFrontEnd.Tools;
using UnigeFrontEnd.Config;
using System.Threading;
using System.Linq;

namespace BabyMind.DAQ
{
	/// <summary>
	/// Class associated to a MCR thread in order to communicate with it
	/// </summary>
	public class MCRThread
	{
		static List<MCRThread> m_list = new List<MCRThread>();

		// used to be filled only once (1st setDeviceAndBoardId call) due to Linux error when calling multiple times the GetAllDevicesBoardId() fct
		static Dictionary<byte, byte> m_deviceIndexBoardId;

		BabyMindLib m_babyMindLib;
		ThreadCommandEnv m_env;

		/// <summary>
		/// Logger associated to this thread
		/// </summary>
		public LoggerWithQueue Log { get; }

		/// <summary>
		/// Get the Last command return string
		/// </summary>
		public string GetLastCommandReturnString { get { return m_env.ReturnString; } }

		/// <summary>
		/// Get the list of the MCR threads
		/// </summary>
		public IReadOnlyList<MCRThread> MCRList { get { return m_list; } }

		/// <summary>
		/// Return total nb of enabled threads
		/// </summary>
		static public int ThreadEnabledNb {
			get
			{
				int l_nb = 0;
				foreach (var l_mcr in m_list)
				{
					if (l_mcr.Enabled)
						l_nb++;
				}
				return l_nb;
			}
		}

		/// <summary>
		/// get the USB device associated to that thread
		/// </summary>
		public int UsbDeviceIndex { get; }

		/// <summary>
		/// return the thread status
		/// </summary>
		public bool Enabled { get; }

		/// <summary>
		/// return the thread index
		/// </summary>
		public byte Index { get; }

		/// <summary>
		/// Associated MCR settings loaded from App Settings file
		/// </summary>
		public AppDataSettings.MCRSettings Settings { get; }

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_settings">MCR Settings from App Settings file</param>
		/// <param name="x_index">MCR index build</param>
		/// <param name="x_appLog">Main application Logger</param>
		/// <param name="x_cmdLoggerPath">Commands Logger Path</param>
		/// <param name="x_filenamePrefixLogger">Prefix used for the logger filename</param>
		/// <param name="x_mcrData">MCR monitoring data</param>
		/// <param name="x_mcrReducedData">MCR monitoring reduced data</param>
		/// <param name="x_errorQueue">Error queue for the logger</param>
		/// <param name="x_queueSize">Limit the size of the strings in the queue</param>
		/// <param name="x_errorThreadEnabled">Error thread is enabled</param>
		/// <param name="x_errorThreadSeverity">Error thread Severity</param>
		public MCRThread(AppDataSettings.MCRSettings x_settings, byte x_index, Logger x_appLog, string x_cmdLoggerPath, string x_filenamePrefixLogger, 
						MonitoringData.Data.MCR x_mcrData, MonitoringData.ReducedData.MCR x_mcrReducedData,
						System.Collections.Concurrent.ConcurrentQueue<string> x_errorQueue,
						int x_queueSize, bool x_errorThreadEnabled, Logger.Severity x_errorThreadSeverity)
		{
			x_appLog.WriteLine($"Preparing Board Environment for Thread index #{x_index} on USB device connected to Board ID #{x_settings.UsbBoardId.BoardIdStart}", Logger.Severity.INFO);
			
			UsbDeviceIndex = -1;
			Index = x_index;
			Enabled = x_settings.Configuration.EnableThread;
			Settings = x_settings;

			if (Enabled)
			{
				string l_logfile = AppDataSettings.EnvironmentSettings.CreateDirAndFile(x_cmdLoggerPath, $"{x_filenamePrefixLogger}{x_index}_");
				Log = new LoggerWithQueue(l_logfile, x_errorQueue, x_queueSize, x_errorThreadEnabled, x_errorThreadSeverity ,true,false);

				buildBoardEnv(x_settings, x_appLog);    // build the environment and initialize BMLib objects
														// set device & board id for MCR0
				if (setDeviceAndBoardId(x_settings.UsbBoardId, x_appLog))
				{
					UsbDeviceIndex = m_babyMindLib.Link.DeviceIndex;

					x_appLog.WriteLine($"Preparing Commands and Thread Environment", Logger.Severity.INFO, x_noTimeStamp:true);

					m_env = new ThreadCommandEnv(m_babyMindLib, Log, x_mcrData, x_mcrReducedData);
					// construct all thread commands
					ThreadCommands l_cmds = new ThreadCommands(m_env);

					x_appLog.WriteLine($"Starting MCRThread{x_index}", Logger.Severity.INFO, x_noTimeStamp: true);

					Thread l_thread = new Thread(new ThreadStart(() => m_env.RunThreadCommands()));
					l_thread.Priority = ThreadPriority.Normal;
					l_thread.Name = $"MCRThread{x_index}";
					l_thread.Start();

					x_appLog.WriteLine($"MCRThread{x_index} started", Logger.Severity.INFO, x_noTimeStamp: true);
				}
				else
					Enabled = false;

				//check that Link and Monitoring board ID are OK
				if (x_mcrData.BoardIDStart != m_babyMindLib.Link.Protocol.Board)
				{
					x_appLog.WriteLine($"Error: Board ID from monitoring data (ID={x_mcrData.BoardIDStart}) is different from Link (ID={m_babyMindLib.Link.Protocol.Board}).\n Disabling Thread...", Logger.Severity.FAILURE, x_noTimeStamp: true);
					Enabled = false;
				}
			}
			else
				x_appLog.WriteLine($"MCRThread{x_index} not enabled", Logger.Severity.INFO, x_noTimeStamp: true);
			m_list.Add(this);
		}

		/// <summary>
		/// Ask from thread to run a command on this thread
		/// </summary>
		///<param name="x_args">argument of the command line, args[0] = name</param>
		///<returns>string message returned in case of error</returns>
		public string RunCommand(string[] x_args)
		{
			if (Enabled)
				return m_env.RunCommand(x_args);
			else
				return $"Thread #{Index} is not enabled. Function '{x_args[0]}' not executed.";
		}

		/// <summary>
		/// Ask from thread to run asynchronously a command on this thread.
		/// ThreadCommandEnv.ResetAsynchronous() must be called 1st
		/// </summary>
		///<param name="x_args">argument of the command line, args[0] = name</param>
		///<returns>string message returned in case of error</returns>
		public string RunAsynchronousCommand(string[] x_args)
		{
			if (Enabled)
				return m_env.RunAsynchronousCommand(x_args);
			else
				return $"Thread #{Index} is not enabled. Function '{x_args[0]}' not executed.";
		}

		#region Board Environment build
		const string m_descFilename = "config-desc.json"; // the hW descriptor

		/// <summary>
		/// Build base environment
		/// </summary>
		/// <param name="x_settings">Configuration Settings for the MCR</param>
		/// <param name="x_log">Main Logger</param>
		void buildBoardEnv(AppDataSettings.MCRSettings x_settings, Logger x_log)
		{
			//------------------------------------------------------------------------------------------------------------
			// Setup auto respond on UserInput redirected to logger
			UserInput l_userInput = new UserInput((x_text, x_severity) =>
				{
					Logger.Severity l_sev;
					switch (x_severity)
					{
						case MessageSeverity.Error: l_sev = Logger.Severity.FAILURE; break;
						case MessageSeverity.Exception: l_sev = Logger.Severity.FAILURE; break;
						case MessageSeverity.Warning: l_sev = Logger.Severity.WARNING; break;
						default: l_sev = Logger.Severity.INFO; break;
					}
					Log.WriteLine(x_text, l_sev);
				}, true);

			l_userInput.FlushPendingException();

			//------------------------------------------------------------------------------------------------------------
			// Register GUI thread invoker
			MainThread l_mainThread = new MainThread(l_action => l_action());			

			//------------------------------------------------------------------------------------------------------------
			// Board Lib settings used by the library
			BoardLibSettings l_settings = new BoardLibSettings();
			l_settings.Configuration.DefaultConfigExtXML = x_settings.Configuration.DefaultConfigExtXML;
			l_settings.Configuration.DefaultSelDevices = x_settings.Configuration.DefaultSelDevices;
			l_settings.Readout.UDPReadout = x_settings.Readout.UDPReadout;
			l_settings.Readout.UDPsettings = new BoardLibSettings.UDPSettings();
			l_settings.Readout.UDPsettings.Port = x_settings.Readout.UDPport;
			l_settings.Readout.UDPsettings.IPaddress = x_settings.Readout.UDP_IPaddress;

			//------------------------------------------------------------------------------------------------------------
			// load JSON description file & data tree
			var l_desc = JsonConverter.LoadConfigDescTree(m_descFilename);
			x_log.WriteLine("Hardware description loaded", Logger.Severity.INFO, x_noTimeStamp: true);

			//------------------------------------------------------------------------------------------------------------
			// create the default data tree from the descriptor
			var l_dataTree = new DataTreeRoot(l_desc);
			x_log.WriteLine("Data tree created", Logger.Severity.INFO, x_noTimeStamp: true);

			//------------------------------------------------------------------------------------------------------------
			// B-MIND properties			
			LibDaqTools.CsvSeparator = ";";
			LibDaqTools.TimeOverflowPeriod = 4000;

			// action executed upon protocol exception : put the msg on the console
			Action<string> l_protocolException = x_msg => l_mainThread.RunOnMainThread(() =>
				Log.WriteLine("Error during Protocol communication (asynchronous cmd/ans) :\n" + x_msg, Logger.Severity.FAILURE));
			// Function creating the implementation of the protocol to use, input = USB, returns a protocol object</param>
			Func<UFE_UnigeFrontEnd.USB.UFE_USB, UFE_UnigeFrontEnd.Protocol.UFE_Protocol> l_protocolCreator
				= x_usb => new Protocol(x_usb, l_protocolException, x_allowAutoReconnect:false);

			//------------------------------------------------------------------------------------------------------------
			//Create Board Link to HW
			var l_link = new LibBoardLink(l_dataTree, l_settings, l_protocolCreator, l_mainThread, l_userInput);
			m_babyMindLib = new BabyMindLib(l_link, new ChainTdmFbwLib(l_link),	x_settings);

			x_log.WriteLine("Board Link loaded", Logger.Severity.INFO, x_noTimeStamp: true);
		}

		/// <summary>
		/// Set device and board ID.
		/// </summary>
		/// <param name="x_settings">USB device and board ID settings</param>
		/// <param name="x_log">Main Logger</param>
		/// <returns>true if set board id and device OK</returns>
		bool setDeviceAndBoardId(AppDataSettings.MCRSettings.UsbBoardIdSettings x_settings, Logger x_log)
		{
			//----------------------------------------------
			// INITAL CODE which doesn't work with LINUX replace by uncommented code
			// get Board ID on all devices connected
			/*m_babyMindLib.Link.GetAllDevicesBoardId();

			string l_str = m_babyMindLib.Link.GetUsbDevices();
			if (l_str == "")
			{
				x_log.WriteLine("No Devices are connected on USB\n", true);
				return false;
			}
			else
				x_log.WriteLine("Devices connected on USB are:\n" + l_str, true);

			try
			{
				m_babyMindLib.Link.SelectUsbDeviceFromBoardId(x_settings.BoardIdStart);
				m_babyMindLib.Link.SetBoardId(x_settings.BoardIdStart);
			}
			catch (Exception x_e)
			{
				x_log.WriteLine($"Error while selecting USB Device Index from BID #{x_settings.BoardIdStart}:" + x_e.Message, true);
				return false;
			*/
			//----------------------------------------------

			//----------------------------------------------
			// PATCH FOR LINUX
			//----------------------------------------------
			// get Board ID on all devices connected
			// only for the 1st time call (Linux issue when called several times)
			if (m_deviceIndexBoardId == null)
			{
				m_babyMindLib.Link.GetAllDevicesBoardId();
				// affect static object which will be used by all Thread constructed later
				m_deviceIndexBoardId = m_babyMindLib.Link.DeviceIndexBoardId;

				// display USB devices only once
				string l_str = m_babyMindLib.Link.GetUsbDevices();
				if (l_str == "")
				{
					x_log.WriteLine("No Devices are connected on USB\n", Logger.Severity.FAILURE, x_noTimeStamp: true);
					return false;
				}
				else
					x_log.WriteLine("Devices connected on USB are:\n" + l_str, Logger.Severity.INFO, x_noTimeStamp: true);
			}else
			{
				// copy to uninitialized AllDevicesBoardId
				m_babyMindLib.Link.SetAllDevicesBoardId(m_deviceIndexBoardId);
			}

			// will find the USB device index from the board id static list
			try
			{
				x_log.WriteLine("Try to get and select USB device index from board ID...", Logger.Severity.INFO, x_noTimeStamp: true);
				byte l_index = m_deviceIndexBoardId.First(x => x.Value == x_settings.BoardIdStart).Key;
				m_babyMindLib.Link.SelectUsbDevice(l_index);
				m_babyMindLib.Link.SetBoardId(x_settings.BoardIdStart);
			}
			catch
			{

				//throw new InvalidOperationException($"The board Id #{x_settings.BoardIdStart} is not found within the board id list of all the connected USB devices");
				x_log.WriteLine($"The board Id #{x_settings.BoardIdStart} is not found within the board id list of all the connected USB devices", Logger.Severity.FAILURE, x_noTimeStamp: true);
				throw new InvalidOperationException("Check that the app-settings correspond to the hardware.");
			}

			//----------------------------------------------
			// END OF PATCH FOR LINUX
			//----------------------------------------------

			if (m_babyMindLib.Link.Protocol.Board != x_settings.BoardIdStart)
			{
				x_log.WriteLine($"Error while checking board ID : Requested BID #{x_settings.BoardIdStart}, connected on BID #{m_babyMindLib.Link.Protocol.Board}", Logger.Severity.FAILURE, x_noTimeStamp: true);
				return false;
			}
			else
			{
				x_log.WriteLine($"Board Link loaded and connected on USB Device Index #{m_babyMindLib.Link.DeviceIndex} with Board ID #{m_babyMindLib.Link.Protocol.Board}", Logger.Severity.INFO, x_noTimeStamp: true);
			}
			return true;
		}

		#endregion
	}
}
