﻿using System;
using System.Text;
using System.Threading;
using UnigeFrontEnd.Tools;
using System.Net.Sockets;

namespace BabyMind.DAQ
{
	public class MonitoringThread
	{
		AppDataSettings.EnvironmentSettings.MonitoringThreadSettings m_setting;
		LoggerWithQueue m_appLog;

		// socket objects
		System.Net.IPAddress m_ip;
		Socket m_listener;
		Socket m_connectedSocket;
		bool m_stop;
		Thread m_thread;

		/// <summary>
		/// Must be initialized before starting the thread...
		/// </summary>
		public TCPIPCommandEnv CommanEnv;
		

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="x_settings">Error Settings from App Settings file</param>
		/// <param name="x_appLog">Main application Logger</param>
		/// <param name="x_ip">IP address</param>
		public MonitoringThread(AppDataSettings.EnvironmentSettings.MonitoringThreadSettings x_settings, LoggerWithQueue x_appLog, System.Net.IPAddress x_ip)
		{
			m_setting = x_settings;
			m_appLog = x_appLog;
			m_ip = x_ip;

			m_appLog.WriteLine($"Building Monitoring Thread", Logger.Severity.INFO);	
		}

		/// <summary>
		/// Stop the socket server
		/// </summary>
		public void Stop()
		{
            try
            {
                if (m_connectedSocket.Connected && m_thread != null) {
                    m_stop = true;
                    m_thread.Abort();
                    m_thread.Join();
                    m_connectedSocket?.Close();
                    m_listener?.Close();
                }
            }
            catch(Exception x_e)
            {
                m_appLog.WriteLine("Stop Monitoring failed from exception (" + x_e.Message + ")", Logger.Severity.CRITICAL);
                return;
            }
		}

		/// <summary>
		/// Start the socket server if not already started
		/// </summary>
		public void Start()
		{
			if(CommanEnv == null)
			{
				m_appLog.WriteLine($"TCPIPCommandEnv is null, Monitoring Thread cannot be started", Logger.Severity.CRITICAL);
				return;
			}

			if ((m_thread == null) || (!m_thread.IsAlive))
			{
				// create the error thread for try to dequeue data and push it to TCP/IP
				m_thread = null;
				m_thread = new Thread(new ThreadStart(() => RunThread()));
				m_thread.Priority = ThreadPriority.BelowNormal;
				m_thread.Name = "MonitoringThread";
				m_thread.Start();
			}
		}

		/// <summary>
		/// Main Core of the thread
		/// </summary>
		void RunThread()
		{
			System.Threading.Thread.Sleep(50);
			m_appLog.WriteLine($"Monitoring Thread started", Logger.Severity.INFO);

			System.Net.IPHostEntry l_ipHostInfo = System.Net.Dns.GetHostEntry(System.Net.Dns.GetHostName());
			System.Net.IPEndPoint l_localEP = new System.Net.IPEndPoint(m_ip, m_setting.IPport);
			m_appLog.WriteLine($"Monitoring Thread Server running on IP Address {l_localEP.Address} / Port {l_localEP.Port}", Logger.Severity.INFO, x_noTimeStamp: true);

			try
			{
				// create the listener socket & bind it to IP/Port
				m_listener = new Socket(l_localEP.Address.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
				m_listener.Bind(l_localEP);
				m_stop = false;
				m_listener.Listen(1); // 1 client allowed per application
			}
			catch (Exception x_e)
			{
				m_appLog.WriteLine("Binding socket to port failed from exception (" + x_e.Message + ")", Logger.Severity.CRITICAL);
				return;
			}
			try
			{
				while (!m_stop)
				{
					m_connectedSocket = m_listener.Accept();
					m_appLog.WriteLine("Monitoring Thread Server connected to remote client", Logger.Severity.INFO);

					while (SocketExtensions.IsConnected(m_connectedSocket) && (!m_stop))
					{						
						byte[] l_bytesToSend = Encoding.UTF8.GetBytes(CommanEnv.RunCommand(new string[] { TCPIPCommands._CMD_MONITORING_DATA }));
						if ((!m_stop) && m_connectedSocket.Connected)
							m_connectedSocket.Send(l_bytesToSend, 0, l_bytesToSend.Length, SocketFlags.None);
						
						// wait for the period
						for (int l_i = 0; l_i < m_setting.PeriodSeconds; l_i++)
						{
							Thread.Sleep(1000);
							Thread.Yield();
							if (!(SocketExtensions.IsConnected(m_connectedSocket)) && (m_stop))
								break;
						}
					}

					// shutdown the socket
					//m_connectedSocket.Shutdown(SocketShutdown.Both);
					m_connectedSocket.Close();
					m_appLog.WriteLine("Monitoring Thread Server Disconnected from remote client (client stop)", Logger.Severity.INFO);
				}
			}
			catch (ThreadAbortException)
			{
				// shutdown the socket
				//m_connectedSocket.Shutdown(SocketShutdown.Both);
				m_connectedSocket.Close();
				m_appLog.WriteLine("Monitoring Thread Server Disconnected from remote client (user stop)", Logger.Severity.INFO);
			}
			catch (Exception x_e)
			{
				m_appLog.WriteLine("Monitoring Thread Server connection acceptance canceled from exception (" + x_e.Message + ")", Logger.Severity.CRITICAL);
				m_listener.Disconnect(true);
				m_connectedSocket.Close();
			}

			m_appLog.WriteLine("Monitoring Thread Server is stopped", Logger.Severity.INFO);
		}
	}
}
