﻿using System;
using BabyMind.Lib;
using System.IO.Ports;
using System.Net;
using System.Net.Sockets;
using System.Text;
using UnigeFrontEnd.Tools;


namespace BabyMind.DAQ
{
	class Program
	{
		public static bool ExitFlag = false;
		public static AppSettings AppSettingsObject;

		static void Main(string[] x_args)
		{
			try
			{
				// disable quick edit mode of console to avoid any stuck or Console.Writeline in case of user copy/edit
				ConsoleWindow.QuickEditMode(false);

				//AppSettings l_appSettingsObject;
				SerialPort l_MCBSerialPort;

				MonitoringData.ReducedData l_monitoringReducedData = new MonitoringData.ReducedData();
				MonitoringData.Data l_monitoringData = new MonitoringData.Data();

				//-------------------------------------------------------------------
				// Load application settings
				Console.WriteLine("DAQ application started...\nReading App settings ...");
				AppSettingsObject = new AppSettings("app-settings.json", "app-settings-defaults.json");

				//-------------------------------------------------------------------
				// create application launch logger
				string l_logfile = AppDataSettings.EnvironmentSettings.CreateDirAndFile(AppSettingsObject.AppSettingsData.Environment.Logger.AppPath,
								AppSettingsObject.AppSettingsData.Environment.Logger.AppDAQPrefix);

				System.Collections.Concurrent.ConcurrentQueue<string> l_appLoggerQueue = new System.Collections.Concurrent.ConcurrentQueue<string>();
				LoggerWithQueue m_log = new LoggerWithQueue(l_logfile, l_appLoggerQueue, 2000, true, AppSettingsObject.AppSettingsData.Environment.ErrorThread.SeverityLevel, true, true);

				string l_versions;
				m_log.WriteLine(TCPIPCommands.Version(out l_versions) + "\n", Logger.Severity.INFO);

				m_log.WriteLine("Starting Configuration...", Logger.Severity.INFO);

				IPAddress l_ipAddress = new IPAddress(AppSettingsObject.AppSettingsData.Environment.Configuration.IPaddress);

				// Building Error thread, log messages inside
				LogErrorThread l_errorThread = new LogErrorThread(AppSettingsObject.AppSettingsData.Environment.ErrorThread, m_log, l_appLoggerQueue,
								AppSettingsObject.AppSettingsData.MCR.Count,
								AppSettingsObject.AppSettingsData.Environment.Logger.ErrorPath,
								AppSettingsObject.AppSettingsData.Environment.Logger.ErrorPrefix,
								l_ipAddress);

				System.Threading.Thread.Sleep(50); //To prevent conflict in accessing the log file

				// Building Monitoring thread, log messages inside
				MonitoringThread l_monitoringThread = new MonitoringThread(AppSettingsObject.AppSettingsData.Environment.MonitoringThread, m_log, l_ipAddress);

				System.Threading.Thread.Sleep(50);
				//------------------------------------
				// build MCR threads and MonitoringData
				m_log.WriteLine("Building MCR Threads...", Logger.Severity.INFO);

				MCRThread l_thread = null;
				for (int l_i = 0; l_i < AppSettingsObject.AppSettingsData.MCR.Count; l_i++)
				{
					//Build DataMonitoring object

					//ReducedData Monitoring
					l_monitoringReducedData.MCRs.Add(new MonitoringData.ReducedData.MCR());

					//Full Data Monitoring
					l_monitoringData.MCRs.Add(new MonitoringData.Data.MCR());

					l_monitoringData.MCRs[l_i].MCR_num = AppSettingsObject.AppSettingsData.MCR[l_i].UsbBoardId.BoardIdStart / 8;
					l_monitoringData.MCRs[l_i].BoardIDStart = AppSettingsObject.AppSettingsData.MCR[l_i].UsbBoardId.BoardIdStart;
					l_monitoringData.MCRs[l_i].BoardIDEnd = AppSettingsObject.AppSettingsData.MCR[l_i].UsbBoardId.BoardIdEnd;
					l_monitoringData.MCRs[l_i].EnableThread = AppSettingsObject.AppSettingsData.MCR[l_i].Configuration.EnableThread;

					for (byte l_b = 0; l_b < l_monitoringData.MCRs[l_i].BoardIDEnd - l_monitoringData.MCRs[l_i].BoardIDStart + 1; l_b++)
					{
						l_monitoringData.MCRs[l_i].Boards.Add(new MonitoringData.Data.Board());
						l_monitoringData.MCRs[l_i].Boards[l_b].BoardID = AppSettingsObject.AppSettingsData.MCR[l_i].UsbBoardId.BoardIdStart + l_b;
					}


					//Build MCR threads
					l_thread = new MCRThread(AppSettingsObject.AppSettingsData.MCR[l_i], (byte)l_i, m_log,
											AppSettingsObject.AppSettingsData.Environment.Logger.CmdPath,
											AppSettingsObject.AppSettingsData.Environment.Logger.ThreadCmdPrefix,
											l_monitoringData.MCRs[l_i], l_monitoringReducedData.MCRs[l_i],
											l_errorThread.QueueArray[l_i],
											AppSettingsObject.AppSettingsData.Environment.ErrorThread.QueueSize,
											AppSettingsObject.AppSettingsData.Environment.ErrorThread.Enabled,
											AppSettingsObject.AppSettingsData.Environment.ErrorThread.SeverityLevel);
				}
				//-------------------------------------------------------------------
				//Open MCB
				m_log.WriteLine("Opening MCB serial port", Logger.Severity.INFO);
				l_MCBSerialPort = MCBLib.MCBOpenPort(AppSettingsObject.AppSettingsData.Environment.Configuration.MCBComPort);
				//-------------------------------------------------------------------
				//Configure MCB
				bool l_internalspill = AppSettingsObject.AppSettingsData.Environment.Configuration.InternalSpill;
				byte l_internalDAQspillWidth = AppSettingsObject.AppSettingsData.Environment.Configuration.InternalDAQPulseWidth;
				byte l_beamSpillWidth = AppSettingsObject.AppSettingsData.Environment.Configuration.BeamDAQPulseWidth;
				MCBLib.MCBConfigure(l_MCBSerialPort, AppSettingsObject.AppSettingsData.Environment.Configuration.MCB_mode, l_internalspill);
				MCBLib.MCBSetInternalDAQPulseWidth(l_MCBSerialPort, l_internalDAQspillWidth);
				MCBLib.MCBSetBeamDAQPulseWidth(l_MCBSerialPort, l_beamSpillWidth);
				
				m_log.WriteLine($"MCB configured as {AppSettingsObject.AppSettingsData.Environment.Configuration.MCB_mode.ToString()}, " +
					$"with internal spill enable: {l_internalspill}, beam DAQ pulse width = {60 + l_beamSpillWidth*4}us and internal DAQ pulse width = {l_internalDAQspillWidth*8} ms " +
					$"on MCB port: {l_MCBSerialPort.PortName}", Logger.Severity.INFO);
				
				//-------------------------------------------------------------------
				// build TCP/IP commands
				m_log.WriteLine("Building TCP/IP commands...", Logger.Severity.INFO);
				// declare a command environment
				TCPIPCommandEnv l_command = new TCPIPCommandEnv(l_thread.MCRList, l_MCBSerialPort, AppSettingsObject.AppSettingsData.Environment.Configuration.MCB_mode, l_internalspill,
																AppSettingsObject.AppSettingsData.Environment.Logger.CmdPath,
																AppSettingsObject.AppSettingsData.Environment.Logger.TCPIPCmdPrefix+"_", 
																l_monitoringData,l_monitoringReducedData,
																l_monitoringThread, AppSettingsObject.AppSettingsData.Environment.MonitoringThread.ReducedMonitoring);
				// construct all TCP IP commands
				TCPIPCommands l_TCPIPCmds = new TCPIPCommands(l_command);

				l_monitoringThread.CommanEnv = l_command;


				//-------------------------------------------------------------------
				// common socket info...
				Socket l_clientSocket = null;
				IPHostEntry l_ipHostInfo = Dns.GetHostEntry(Dns.GetHostName());		

				//-------------------------------------------------------------------
				// Create the Force Disconnect Thread
				IPEndPoint l_localForceDisconnect = new IPEndPoint(l_ipAddress, AppSettingsObject.AppSettingsData.Environment.Configuration.ForceDisconnectIPport);
				m_log.WriteLine($"Opening TCP socket for disconnection on Host Name: {Dns.GetHostName()}, IP Address: {l_localForceDisconnect.Address}, Port: {l_localForceDisconnect.Port}", Logger.Severity.INFO);
				
				System.Threading.Thread l_forceDisconnectThread = new System.Threading.Thread(new System.Threading.ThreadStart(() =>
				{
					Socket l_forceDisconnectSock = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
					l_forceDisconnectSock.Bind(l_localForceDisconnect);
					l_forceDisconnectSock.Listen(5);

					m_log.WriteLine($"Disconnection socket is waiting for a client in a separated thread...", Logger.Severity.INFO);
					while(!ExitFlag)
					{
						//Accept Client
						Socket l_cs = l_forceDisconnectSock.Accept();
						m_log.WriteLine($"Disconnection Client connected from IP: {l_cs.RemoteEndPoint}", Logger.Severity.INFO);
						//---------------
						//Receive Commands in a while loop
						byte[] l_buffer = new byte[100];
						int l_b = l_cs.Receive(l_buffer);
						string l_req = "";
						l_req += Encoding.ASCII.GetString(l_buffer, 0, l_b);
						if((!String.IsNullOrEmpty(l_req)) && (l_req == AppSettingsObject.AppSettingsData.Environment.Configuration.ForceDisconnectPassword))
						{
							if ((l_clientSocket != null) && l_clientSocket.Connected)
								l_clientSocket.Close();
							m_log.WriteLine($"Disconnection request from client accepted", Logger.Severity.INFO);
						}else
							m_log.WriteLine($"Disconnection request from client rejected (wrong password)", Logger.Severity.WARNING);
						
						l_cs.Close();
						m_log.WriteLine($"Disconnection client disconnected from server", Logger.Severity.INFO);
					}
				}))
				{
					Priority = System.Threading.ThreadPriority.Normal,
					Name = "DisconnectThread"
				};
				l_forceDisconnectThread.Start();

				//-------------------------------------------------------------------
				// Open the TCPIP Socket
				IPEndPoint l_localEP = new IPEndPoint(l_ipAddress, AppSettingsObject.AppSettingsData.Environment.Configuration.IPport);
				m_log.WriteLine($"Opening TCP socket on Host Name: {Dns.GetHostName()}, IP Address: {l_localEP.Address}, Port: {l_localEP.Port}", Logger.Severity.INFO);							
				
				Socket l_sock = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
				l_sock.Bind(l_localEP);
				l_sock.Listen(5);
				m_log.WriteLine($"Socket is open, waiting for a client connection...", Logger.Severity.INFO);
				

				while (!ExitFlag)
				{
					try
					{
						//Accept Client
						l_clientSocket = l_sock.Accept();						
						m_log.WriteLine($"Client connected from IP: {l_clientSocket.RemoteEndPoint}", Logger.Severity.INFO);
						l_clientSocket.Send(Encoding.ASCII.GetBytes("Server acknowledges client connection"));
						//---------------
						//Receive Commands in a while loop
						byte[] l_buffer = new byte[100000];
						string l_req = null;

						while (!ExitFlag) 
						{
							
							int l_b = l_clientSocket.Receive(l_buffer);
							l_req += Encoding.ASCII.GetString(l_buffer, 0, l_b);
							m_log.WriteLine("" + l_req, Logger.Severity.INFO);

							string[] l_str = l_req.Split(' ');
							string l_ans = l_command.RunCommand(l_str); //Run command and receive answer
							m_log.WriteLine(l_ans, Logger.Severity.INFO);
							l_clientSocket.Send(Encoding.ASCII.GetBytes(l_ans));

							//reset
							l_req = null;
							Array.Clear(l_buffer, 0, l_buffer.Length);
						}
					}
					catch (SocketException)
					{
						m_log.WriteLine($"Client disconnected, waiting for a new client connection...", Logger.Severity.INFO);						
					}
					catch (Exception x_e)
					{
						m_log.WriteLine($"Exception caught in TCP listening loop, msg=" + x_e.Message, Logger.Severity.WARNING);
					}
				}

				//-------------------------------------------------------------------				
				//Close serial
				m_log.WriteLine("Closing the serial port...", Logger.Severity.INFO);
				l_MCBSerialPort.Close();

				//Close socket
				m_log.WriteLine("Closing the socket...", Logger.Severity.INFO);
				l_sock.Close();

				m_log.WriteLine("Application ended...", Logger.Severity.INFO);

			}
			catch(Exception x_e)
			{
				Console.WriteLine(x_e.Message);
			}
			Environment.Exit(0);
		}
		
	}
	
}
