﻿#define ASYNCHRONOUS_MONITORING

using System;
using System.Collections.Generic;
using System.Threading;

namespace BabyMind.DAQ
{
	/// <summary>
	/// TCP/IP commands
	/// </summary>
	class TCPIPCommands
	{
		public static readonly string _CMD_HELP = "Help";
		public static readonly string _CMD_EXIT = "Exit";
		public static readonly string _CMD_ALL_FIRMWARE = "AllFirmware";
		public static readonly string _CMD_ALL_STATUS = "AllStatus";
		public static readonly string _CMD_ALL_SYNC = "AllSYNC";
		public static readonly string _CMD_ALL_PREPARE_DAQ = "AllPrepareDAQ";
		public static readonly string _CMD_ALL_STOP_DAQ = "AllStopDAQ";
		public static readonly string _CMD_START_TRIGGER = "StartTrigger";
		public static readonly string _CMD_STOP_TRIGGER = "StopTrigger";
		public static readonly string _CMD_ALL_MONITORING = "AllMonitoring";
		public static readonly string _CMD_MONITORING_ONE_BOARD_ONE_CH = "MonitoringOneBoardOneCh";
		public static readonly string _CMD_ALL_RUN_STATUS = "AllRunStatus";
		public static readonly string _CMD_VARIABLE = "GetVariable";
		public static readonly string _CMD_DATA_RATE = "GetDataRate";
		public static readonly string _CMD_MAX_RATE = "GetMaxRate";
		public static readonly string _CMD_TIME = "GetTimeElapsed";
		public static readonly string _CMD_FILE_NAME = "GetFileName";
		public static readonly string _CMD_FILE_SIZE = "GetFileSize";
		public static readonly string _CMD_SPILL_NUMBER = "GetSpillNumber";
		public static readonly string _CMD_VERSION = "Version";
		public static readonly string _CMD_APP_SETTINGS = "GetAppSettings";
		public static readonly string _CMD_SET_RUN_PROPERTIES = "SetRunProperties";
		public static readonly string _CMD_MONITORING_DATA = "GetMonitoringData";
		public static readonly string _CMD_START_STOP_MONITORING_DATA = "StartStopMonitoringData";


		/// <summary>
		/// Constructor of all TCP/IP commands
		/// Add here all commands you have defined and which will be interpreted/executed from TCP/IP sockets
		/// </summary>
		/// <param name="x_command">Command object used to associate with</param>
		public TCPIPCommands(TCPIPCommandEnv x_command)
		{
			HelpCmd _HelpCmd = new HelpCmd(x_command);
			ExitCmd _ExitCmd = new ExitCmd(x_command);

			GetAllStatus _GetAllStatus = new GetAllStatus(x_command);
			AllPrepareDAQ _AllPrepareDAQ = new AllPrepareDAQ(x_command);
			AllStopDAQ _AllStopDAQ = new AllStopDAQ(x_command);
			StartTriggerCmd _StartTriggerCmd = new StartTriggerCmd(x_command);
			StopTriggerCmd _StopTriggerCmd = new StopTriggerCmd(x_command);
			AllMonitoring _AllMonitoring = new AllMonitoring(x_command);
			MonitoringOneBoardOneCh _MonitoringOneBoardOneCh = new MonitoringOneBoardOneCh(x_command);
			AllRunStatus _AllRunStatus = new AllRunStatus(x_command);
			GetDataRate _GetDataRate = new GetDataRate(x_command);
			GetMaxRate _GetMaxRate = new GetMaxRate(x_command);
			GetTimeElapsed _GetTimeElapsed = new GetTimeElapsed(x_command);
			GetFileSize _GetFileSize = new GetFileSize(x_command);
			GetFileName _GetFileName = new GetFileName(x_command);
			GetVariable _GetVariable = new GetVariable(x_command);
			GetSpillNumber _GetSpillNumber = new GetSpillNumber(x_command);
			GetFirmwareCmd _GetFirmwareCmd = new GetFirmwareCmd(x_command);
			GetVersionCmd _GetVesionCmd = new GetVersionCmd(x_command);
			GetAppSettings _GetAppSettings = new GetAppSettings(x_command);
			SetRunProperties _SetRunProperties = new SetRunProperties(x_command);
			GetMonitoringData _GetMonitoringData = new GetMonitoringData(x_command);
			StartStopMonitoringDataCmd _StartStopMonitoringDataCmd = new StartStopMonitoringDataCmd(x_command);
			AllSync _AllSync = new AllSync(x_command);
		}

		/// <summary>
		/// Help
		/// </summary>
		class HelpCmd : GenericCommand
		{
			public HelpCmd(CommandEnv x_command) : base(x_command, _CMD_HELP) { }

			public override string Execute(string[] x_args)
			{
				return m_command.Help();
			}
			public override string help()
			{
				return "Help on the command interpreter";
			}
		}

		/// <summary>
		/// Exit
		/// </summary>
		public class ExitCmd : TCPIPCommand
		{
			public ExitCmd(TCPIPCommandEnv x_command) : base(x_command, _CMD_EXIT) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					var l_args = new string[] { ThreadCommandEnv._CMD_EXIT };
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					Program.ExitFlag = true;
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Exit the DAQ program on DAQ PC.";
			}
		}
		
//////////////////////////////////
		/// <summary>
		/// Status
		/// </summary>
		public class GetAllStatus : TCPIPCommand
		{
			public GetAllStatus(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_STATUS) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					var l_args = new string[] { ThreadCommands._CMD_BOARD_STATUS };
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}				
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Return the status parameters";
			}
		}

		public class AllSync : TCPIPCommand
		{
			public AllSync(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_SYNC) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					var l_args = new string[] { ThreadCommands._CMD_SYNC };
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Return the status parameters";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// AllPrepareDAQ
		/// </summary>
		public class AllPrepareDAQ : TCPIPCommand
		{
			public AllPrepareDAQ(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_PREPARE_DAQ) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					
					var l_args = new string[] { ThreadCommands._CMD_PREPARE_DAQ };
					string l_str = "";

#if ASYNCHRONOUS_MONITORING
					ThreadCommandEnv.ResetAsynchronous(MCRThread.ThreadEnabledNb);
#endif
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
#if ASYNCHRONOUS_MONITORING
							l_mcr.RunAsynchronousCommand(l_args);
#else
							l_str += l_mcr.RunCommand(l_args);
#endif
					}

#if ASYNCHRONOUS_MONITORING
					ThreadCommandEnv.WaitAsynchronous();
					//Get return strings
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.GetLastCommandReturnString;
					}					
#endif
					return (l_str);

					}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "PrepareDAQ for all boards";
			}
		}

		//////////////////////////////////
		/// <summary>
		/// AllStopDAQ
		/// </summary>
		public class AllStopDAQ : TCPIPCommand
		{
			public AllStopDAQ(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_STOP_DAQ) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					var l_args = new string[] { ThreadCommands._CMD_STOP_DAQ };
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Stop DAQ for all MCRs";
			}
		}

		//////////////////////////////////
		/// <summary>
		/// StartTrigger
		/// </summary>
		public class StartTriggerCmd : TCPIPCommand
		{
			public StartTriggerCmd(TCPIPCommandEnv x_command) : base(x_command, _CMD_START_TRIGGER) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					//Lib.MCBLib.MCBConfigure(m_command.MCBSerialPort, m_command.MCBMode, m_command.EnableInternalSpill);
					Lib.MCBLib.MCBReadout(m_command.MCBSerialPort, true, true, true);				
					return "Trigger is started. Taking data...";
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: StartTrigger. MCB StartTrigger.";
			}
		}

		//////////////////////////////////
		/// <summary>
		/// Stop Trigger
		/// </summary>
		public class StopTriggerCmd : TCPIPCommand
		{
			public StopTriggerCmd(TCPIPCommandEnv x_command) : base(x_command, _CMD_STOP_TRIGGER) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					Lib.MCBLib.MCBReadout(m_command.MCBSerialPort, false, false, false);
					return "Trigger is stopped.";
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: StopTrigger.";
			}
		}


		//////////////////////////////////
		/// <summary>
		/// Get Spill Number
		/// </summary>
		public class GetSpillNumber : TCPIPCommand
		{
			public GetSpillNumber(TCPIPCommandEnv x_command) : base(x_command, _CMD_SPILL_NUMBER) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = "";
					l_str += "MCB mode is: " + m_command.MCBMode.ToString()+"\n";
					l_str += "Spill Number is : " + Lib.MCBLib.MCBGetSpillNb(m_command.MCBSerialPort) +"\n";
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: Get Spill Number.";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// AllMonitoring
		/// </summary>
		public class AllMonitoring : TCPIPCommand
		{
			public AllMonitoring(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_MONITORING) { }

			public override string Execute(string[] x_args)
			{
				try
				{

					var l_args = new string[] { ThreadCommands._CMD_MONITORING };
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{

						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get slow control values for all boards";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// Get Monitoring Data
		/// </summary>
		public class GetMonitoringData : TCPIPCommand
		{
			byte[] m_errors;
			const byte _ErrorLevel = 2;
			bool[] m_enabled;
			bool m_reducedMonitoring;
			int m_nbEnabled = 0;

			public GetMonitoringData(TCPIPCommandEnv x_command) : base(x_command, _CMD_MONITORING_DATA)
			{
				DisableLog = true;
				m_reducedMonitoring = m_command.ReducedMonitoringEnabled;
				m_errors = new byte[m_command.MCRList.Count];
				m_enabled = new bool[m_command.MCRList.Count];
				clearErrors();
			}	
			
			/// <summary>
			/// Clear errors
			/// </summary>
			void clearErrors()
			{
				m_nbEnabled = 0;

				for (int l_i = 0; l_i < m_command.MCRList.Count; l_i++)
				{
					m_errors[l_i] = 0;
					m_enabled[l_i] = m_command.MCRList[l_i].Enabled;
					if (m_enabled[l_i])
						m_nbEnabled++;
				}
				m_reducedMonitoring = m_command.ReducedMonitoringEnabled;
			}

			/// <summary>
			/// Execute
			/// </summary>
			/// <param name="x_args"></param>
			/// <returns></returns>
			public override string Execute(string[] x_args)
			{
				try
				{					
					if ((x_args.Length > 1) && (x_args[1] == "/c"))
						clearErrors();

					string[] l_args = new string[2];
					l_args[0] = ThreadCommands._CMD_MONITORING_DATA;                    
					l_args[1] = m_reducedMonitoring ? "/r" : "/f";

#if ASYNCHRONOUS_MONITORING
					if (m_nbEnabled>0)
						ThreadCommandEnv.ResetAsynchronous(m_nbEnabled);
#endif					
					//Loop over all MCR 
					for (int l_i = 0; l_i < m_command.MCRList.Count; l_i++)
					{
						if (m_enabled[l_i])
						{
#if ASYNCHRONOUS_MONITORING
							m_command.MCRList[l_i].RunAsynchronousCommand(l_args); //Will trigger the thread to execute the command asynchronously
#else
							m_command.MCRList[l_i].RunCommand(l_args); //Will synchronously return Done but also fill the m_command.MonitorData structure
#endif
						}						
					}

#if ASYNCHRONOUS_MONITORING
					if (m_nbEnabled > 0)
						ThreadCommandEnv.WaitAsynchronous();
#endif

					//Loop over all MCR to get strings
					for (int l_i = 0; l_i < m_command.MCRList.Count; l_i++)
					{
						if (m_enabled[l_i])
						{
							if (m_errors[l_i] < _ErrorLevel)
							{
								if (m_command.MCRList[l_i].GetLastCommandReturnString.StartsWith(GenericCommand._ERR_RETURN))
								{
									m_errors[l_i]++;
									if ((m_errors[l_i] == _ErrorLevel) && (!m_reducedMonitoring))
									{
										m_errors[l_i] = 0;
										m_reducedMonitoring = true;
										m_command.Log.WriteLine($"Switch to reduced Monitoring Data on all Threads due to {_ErrorLevel} successive errors on thread #{l_i}", UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
									}
								}
								else
									m_errors[l_i] = 0;
							}
							else if (m_errors[l_i] == _ErrorLevel) // log message once and disable monitoring
							{								
								m_errors[l_i]++;
								m_enabled[l_i] = false;
								if (m_nbEnabled > 0)
									m_nbEnabled--;

								m_command.Log.WriteLine($"Disable Get Monitoring Data on Thread #{l_i} due to {_ErrorLevel} successive errors", UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
								
							}
						}
					}

					m_command.MonitoringData.MCB.MCBMode = Lib.MCBLib.MCBDAQmodes.EXT_IN_NIM1.ToString();
					m_command.MonitoringData.MCB.SpillNumber = Lib.MCBLib.MCBGetSpillNb(m_command.MCBSerialPort);	

					//Reduced data
					m_command.MonitoringReducedData.MCB.SpillNumber = Lib.MCBLib.MCBGetSpillNb(m_command.MCBSerialPort);

					if (m_reducedMonitoring)
					{
						return (MonitoringData.WriteReducedDataToString(m_command.MonitoringReducedData) + "\n");
					}else
					{
						return (MonitoringData.WriteDataToString(m_command.MonitoringData) + "\n");
					}    
					
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Help: GetVariable MCR# BID# VARIABLE_CHANNEL#";
			}
		}

		//////////////////////////////////


		//////////////////////////////////
		/// <summary>
		/// Get Variable one channel
		/// </summary>
		public class GetVariable : TCPIPCommand
		{
			public GetVariable(TCPIPCommandEnv x_command) : base(x_command, _CMD_VARIABLE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = x_args;
					l_args[0] = ThreadCommands._CMD_VARIABLE;

					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart / 8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Help: GetVariable MCR# BID# VARIABLE_CHANNEL#";
			}
		}

		//////////////////////////////////
		/// <summary>
		/// MonitoringOneBoardOneCh
		/// </summary>
		public class MonitoringOneBoardOneCh : TCPIPCommand
		{
			public MonitoringOneBoardOneCh(TCPIPCommandEnv x_command) : base(x_command, _CMD_MONITORING_ONE_BOARD_ONE_CH) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = x_args;
					l_args[0] = ThreadCommands._CMD_MONITORING_ONE_CH;

					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart /8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get slow control values for one board one ch";
			}
		}


		//////////////////////////////////
		/// <summary>
		/// Get DataRate
		/// </summary>
		/// 
		public class GetDataRate : TCPIPCommand
		{
			public GetDataRate(TCPIPCommandEnv x_command) : base(x_command, _CMD_DATA_RATE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = { ThreadCommands._CMD_DATA_RATE };


					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart / 8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))
							
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return " Help: Get Data Rate MCR#";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// Get Max Rate
		/// </summary>
		/// 
		public class GetMaxRate : TCPIPCommand
		{
			public GetMaxRate(TCPIPCommandEnv x_command) : base(x_command, _CMD_MAX_RATE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = x_args;
					l_args[0] = ThreadCommands._CMD_MAX_RATE;


					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart / 8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))

							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get Max rate";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// Get Elapsed Time
		/// </summary>
		/// 
		public class GetTimeElapsed : TCPIPCommand
		{
			public GetTimeElapsed(TCPIPCommandEnv x_command) : base(x_command, _CMD_TIME) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = x_args;
					l_args[0] = ThreadCommands._CMD_TIME;


					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart / 8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))

							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get Time";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// Get File name
		/// </summary>
		/// 
		public class GetFileName : TCPIPCommand
		{
			public GetFileName(TCPIPCommandEnv x_command) : base(x_command, _CMD_FILE_NAME) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = x_args;
					l_args[0] = ThreadCommands._CMD_FILE_NAME;


					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart / 8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))

							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get File name";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// Get Elapsed Time
		/// </summary>
		/// 
		public class GetFileSize : TCPIPCommand
		{
			public GetFileSize(TCPIPCommandEnv x_command) : base(x_command, _CMD_FILE_SIZE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = { ThreadCommands._CMD_FILE_SIZE };

					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						int l_mcr_num = (int)l_mcr.Settings.UsbBoardId.BoardIdStart / 8;

						if (l_mcr.Enabled && (l_mcr_num.ToString() == x_args[1]))

							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get File size";
			}
		}
		//////////////////////////////////
		/// <summary>
		/// Get Run status from all boards
		/// </summary>
		public class AllRunStatus : TCPIPCommand
		{
			public AllRunStatus(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_RUN_STATUS) { }

			public override string Execute(string[] x_args)
			{
				try
				{

					var l_args = new string[] { ThreadCommands._CMD_RUN_STATUS };
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{

						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get run status for all boards";
			}
		}

		/// <summary>
		/// Start/Stop Monitoring Data
		/// </summary>
		public class StartStopMonitoringDataCmd : TCPIPCommand
		{
			public StartStopMonitoringDataCmd(TCPIPCommandEnv x_command) : base(x_command, _CMD_START_STOP_MONITORING_DATA) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = "Unknown command or parameter";
					if (x_args.Length > 1)
					{
						if (x_args[1] == "/start")
						{
							m_command.MonitoringThread.Start();
							l_str = "Monitoring Data started";
						}
						else if (x_args[1] == "/stop")
						{
							m_command.MonitoringThread.Stop();
							l_str = "Monitoring Data stopped";
						}
					}
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Start/stop the monitoring Data \n  parameters: /stop or /start";
			}
		}

		
		/// <summary>
		/// Get Version of the DAQ software
		/// </summary>
		public class GetVersionCmd : TCPIPCommand
		{
			public GetVersionCmd(TCPIPCommandEnv x_command) : base(x_command, _CMD_VERSION) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_libVersions;
					string l_str = Version(out l_libVersions);
					if ((x_args.Length > 1) && (x_args[1] == "/f"))
						l_str += "\n" + l_libVersions;
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Get the software version(s)\n  /f:Optional libraries version";
			}
		}

		/// <summary>
		/// Return software version of the application
		/// </summary>
		/// <param name="x_libVersions">full libraries versions</param>
		/// <returns>Software version</returns>
		public static string Version(out string x_libVersions)
		{
			int l_latestSvnReleased = 0;
			x_libVersions = UnigeFrontEnd.Tools.Utils.GetLibAssemblyVersion(out l_latestSvnReleased);
			UnigeFrontEnd.Tools.Utils.AppAssemblyInfoId l_info = UnigeFrontEnd.Tools.Utils.GetAppAssemblyInfo(System.Reflection.Assembly.GetExecutingAssembly().Location);

			return "Software Version: " + l_info.Version + $"({l_latestSvnReleased})";
		}

		//////////////////////////////////
		/// <summary>
		/// Get Firmware
		/// </summary>
		public class GetFirmwareCmd : TCPIPCommand
		{
			public GetFirmwareCmd(TCPIPCommandEnv x_command) : base(x_command, _CMD_ALL_FIRMWARE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args;
					bool l_full = ((x_args.Length > 1) && (x_args[1] == "/f"));
					if (l_full)
						l_args = new string[] { ThreadCommands._CMD_FIRMWARE, "/f" };
					else
						l_args = new string[] { ThreadCommands._CMD_FIRMWARE };

					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return " Usage: FirmwareVersion [/f].\n  Get the firmware versions\n  /f:Optional full text version";
			}
		}

		
		//////////////////////////////////
		/// <summary>
		/// Get Application Settings
		/// </summary>
		public class GetAppSettings : TCPIPCommand
		{
			public GetAppSettings(TCPIPCommandEnv x_command) : base(x_command, _CMD_APP_SETTINGS) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					return Program.AppSettingsObject.AppSettingsData.DisplayAllVariables();
					
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return " Usage: GetAppSettings.\n  List Application settings";
			}
		}

		/// <summary>
		/// Set Run Properties
		/// </summary>
		public class SetRunProperties :TCPIPCommand
		{
			public SetRunProperties(TCPIPCommandEnv x_command) : base(x_command, _CMD_SET_RUN_PROPERTIES) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string[] l_args = x_args;
					l_args[0] = ThreadCommands._CMD_SET_RUN_PROP;

				
					string l_str = "";
					//Loop over all MCR 
					foreach (var l_mcr in m_command.MCRList)
					{
						if (l_mcr.Enabled)
							l_str += l_mcr.RunCommand(l_args);
					}
					return (l_str);
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: SetRunProperties {RunNumber} {RunType}. ";
			}
		}
	}

}
