﻿using System;
using System.Threading;
using System.IO;

namespace BabyMind.DAQ
{
	/// <summary>
	/// Thread Commands
	/// </summary>
	class ThreadCommands
	{
		public static readonly string _CMD_FIRMWARE = "Firmware";
		public static readonly string _CMD_BOARD_STATUS = "BoardStatus";
		public static readonly string _CMD_VARIABLE = "Variable";
		public static readonly string _CMD_SYNC = "SYNC";
		public static readonly string _CMD_PREPARE_DAQ = "PrepareDAQ";
		public static readonly string _CMD_STOP_DAQ = "StopDAQ";
		public static readonly string _CMD_MONITORING = "Monitoring";
		public static readonly string _CMD_MONITORING_ONE_CH = "MonitoringOneCh";
		public static readonly string _CMD_RUN_STATUS= "RunStatus";
		public static readonly string _CMD_DATA_RATE = "DataRate";
		public static readonly string _CMD_MAX_RATE = "MaxRate";
		public static readonly string _CMD_TIME = "TimeElapsed";
		public static readonly string _CMD_FILE_NAME = "FileName";
		public static readonly string _CMD_FILE_SIZE = "FileSize";
		public static readonly string _CMD_SET_RUN_PROP = "SetRunProp";
		public static readonly string _CMD_MONITORING_DATA = "MonitoringData";
		public static readonly string _CMD_RESET_FIFO = "ResetFifo";

		/// <summary>
		/// Constructor of all TCP/IP commands
		/// Add here all commands you have defined and which will be interpreted/executed from TCP/IP sockets
		/// </summary>
		/// <param name="x_command">Command object used to associate with</param>
		public ThreadCommands(ThreadCommandEnv x_command)
		{
			ThreadCommandEnv.Exit _Exit = new ThreadCommandEnv.Exit(x_command);
			BoardFirmware _BoardFirmware = new BoardFirmware(x_command);
			BoardStatus _BoardStatus = new BoardStatus(x_command);
			GetVariable _GetVariable = new GetVariable(x_command);
			PrepareDAQCmd _PrepareDAQ = new PrepareDAQCmd(x_command);
			StopDAQCmd _StopDAQ = new StopDAQCmd(x_command);
			Monitoring _Monitoring = new Monitoring(x_command);
			MonitoringOneCh _MonitoringOneCh = new MonitoringOneCh(x_command);
			RunStatusCmd _RunStatusCmd = new RunStatusCmd(x_command);
			DataRate _DataRate = new DataRate(x_command);
			MaxRate _MaxRate = new MaxRate(x_command);
			TimeElapsed _TimeElapsed = new TimeElapsed(x_command);
			DataFileName _DataFileName = new DataFileName(x_command);
			FileSize _FileSize = new FileSize(x_command);
			SetRunPropCmd _SetRunPropCmd = new SetRunPropCmd(x_command);
			GetMonitoringData _GetMonitoringData = new GetMonitoringData(x_command);
			ResetFifo _ResetFifo = new ResetFifo(x_command);
			Sync _Sync = new Sync(x_command);
		}

		/// <summary>
		/// Monitoring Data
		/// </summary>
		public class GetMonitoringData : ThreadCommand
		{
			public GetMonitoringData(ThreadCommandEnv x_command) : base(x_command, _CMD_MONITORING_DATA) { DisableLog = true; }

			public override string Execute(string[] x_args)
			{
				string l_str = "";
				try
				{
					
					//reduce data
					if ((x_args.Length>1) && (x_args[1] == "/r"))
					{
						//Fill Monitoring ReducedData
						m_command.MCR_reducedData.File_size = (int)m_command.BMLib.Link.XferBytes / 1000;
						m_command.MCR_reducedData.IsTransferingData = m_command.BMLib.Link.IsTransferingData;
					}                    	
					else // full data
					{
						//Fill Monitoring data MCR related values 
						m_command.MCR_data.File_name = m_command.BMLib.MCRSettings.FullName;
						m_command.MCR_data.File_size = (int)m_command.BMLib.Link.XferBytes / 1000;
						m_command.MCR_data.Elapsed_time = m_command.BMLib.Link.ElapsedTime.ToString(@"dd\.hh\:mm\:ss");
						m_command.MCR_data.Avg_rate = (int)m_command.BMLib.Link.AvgXferRate;
						m_command.MCR_data.Max_rate = (int)m_command.BMLib.Link.MaxXferRate;
						m_command.MCR_reducedData.IsTransferingData = m_command.BMLib.Link.IsTransferingData;

						//Loop for boards in MCR
						for (int l_b = 0; l_b < m_command.MCR_data.Boards.Count; l_b++)
						{
							m_command.BMLib.Link.SetBoardId((byte)m_command.MCR_data.Boards[l_b].BoardID);

							// ASIC output has a bug so temp. read is wrong 
							//m_command.MCR_data.Boards[l_b].ASIC0_temp = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.TEMP_ASIC0), 2));
							//m_command.MCR_data.Boards[l_b].ASIC1_temp = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.TEMP_ASIC1), 2));
							//m_command.MCR_data.Boards[l_b].ASIC2_temp = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.TEMP_ASIC2), 2));
							m_command.MCR_data.Boards[l_b].FPGA_temp = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.TEMP_FPGA), 2));
							m_command.MCR_data.Boards[l_b].Board_temp = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.TEMP_BOARD), 2));
							m_command.MCR_data.Boards[l_b].Humidity = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.HR_BOARD), 2));
							m_command.MCR_data.Boards[l_b].HV = Convert.ToSingle(Math.Round(m_command.BMLib.Link.GetHouseKeepingValue(Lib.LibBoardLink.HouseKeepingChannel.HV), 2));

							m_command.BMLib.Link.ReadStatus();
							m_command.MCR_data.Boards[l_b].L0FifoErr = m_command.BMLib.Link.GetBoolVariable("Board.DirectParam.L0FifoReset");
							m_command.MCR_data.Boards[l_b].L1FifoErr = m_command.BMLib.Link.GetBoolVariable("Board.DirectParam.L1FifoReset");
							m_command.MCR_data.Boards[l_b].L2FifoErr = m_command.BMLib.Link.GetBoolVariable("Board.DirectParam.L2FifoReset");

							//Raise Error if Err counters are different from previous values
							if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST) != m_command.MCR_data.Boards[l_b].FSYNC_Err)
								m_command.Log.WriteLine("FSYNC_LOST: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
							if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST) != m_command.MCR_data.Boards[l_b].GTRIG_LOST)
								m_command.Log.WriteLine("GTRIG_LOST: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
							if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED) != m_command.MCR_data.Boards[l_b].PLL_LOCKED)
								m_command.Log.WriteLine("PLL_LOCKED: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
							if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF) != m_command.MCR_data.Boards[l_b].PARITY_EOF)
								m_command.Log.WriteLine("PARITY_EOF: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);


							m_command.MCR_data.Boards[l_b].FSYNC_Err = m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST);
							m_command.MCR_data.Boards[l_b].GTRIG_LOST = m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST);
							m_command.MCR_data.Boards[l_b].PLL_LOCKED = m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED);
							m_command.MCR_data.Boards[l_b].PARITY_EOF = m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF);

							
							
						}
					}
					l_str += "Done";
					
				}
				catch (Exception x_e)
				{
					for (int l_b = 0; l_b < m_command.MCR_data.Boards.Count; l_b++)
					{
						m_command.MCR_data.Boards[l_b].Clear();
					}
					l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);

					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
				return (l_str);
			}
			public override string help()
			{
				return " Usage: MonitoringData for the thread MCR. ";
			}
		}



		/// <summary>
		/// Firmware
		/// </summary>
		public class BoardFirmware : ThreadCommand
		{
			public BoardFirmware(ThreadCommandEnv x_command) : base(x_command, _CMD_FIRMWARE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					//Loop for boards in MCR
					string l_str = "";

					if (!m_command.BMLib.Link.ReadStatus())
					{
						return "Cannot access MCR "+ m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart /8 +" \n";
					}
					for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart;
							l_i <= m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd; l_i++)
					{
						m_command.BMLib.Link.SetBoardId(l_i);
						l_str += "Board ID: " + l_i + "\n";

						string l_warning;
						bool l_full = ((x_args.Length > 1) && (x_args[1] == "/f"));
						l_str += (m_command.BMLib.Link.GetFirmwareVersion(out l_warning, l_full));
						l_str += l_warning;
                        l_str += "\n";


                    }

					return (l_str);

				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return " Usage: FirmwareVersion [/f].\n  Get the firmware versions\n  /f:Optional full text version";
			}
		}


		/// <summary>
		/// Status
		/// </summary>
		public class BoardStatus : ThreadCommand
		{
			public BoardStatus(ThreadCommandEnv x_command) : base(x_command, _CMD_BOARD_STATUS) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					//Loop for boards in MCR
					string l_str = "";
					for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart;
							l_i <= m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd; l_i++)
					{
						m_command.BMLib.Link.SetBoardId(l_i);
						l_str += "\nBoard ID: " + l_i + "\n";
						
						if (m_command.BMLib.Link.ReadStatus())
						{
							l_str += m_command.BMLib.Link.Board.StatusParameters.GetValues();
						}
						else
							l_str += "Cannot access MCR" + (m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart / 8)+ " \n";

					}

					return (l_str);

				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Return the status parameters";
			}
		}
		/// <summary>
		/// Get Variable
		/// </summary>
		public class GetVariable : ThreadCommand
		{
			public GetVariable(ThreadCommandEnv x_command) : base(x_command, _CMD_VARIABLE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					//Loop for boards in MCR
					string l_str = "";
					string l_parameter_path = "";
					bool l_BID_found = false;

					for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart;
							l_i <= m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd; l_i++)
					{
						// x_args[1] = MCR#, x_args[2] = BID#, x_args[3] = Channel

						if (l_i.ToString() == x_args[2])
						{
							l_BID_found = true;
							if (x_args[3] == 0.ToString()) l_parameter_path = "Board.DirectParam.L0FifoReset";
							if (x_args[3] == 1.ToString()) l_parameter_path = "Board.DirectParam.L1FifoReset";
							if (x_args[3] == 2.ToString()) l_parameter_path = "Board.DirectParam.L2FifoReset";

							m_command.BMLib.Link.SetBoardId(l_i);

							m_command.BMLib.Link.ReadStatus();
							l_str += m_command.BMLib.Link.GetBoolVariable(l_parameter_path).ToString();
						}
					}

					if(l_BID_found == false)
					{
						l_str += "BID was not found !!";
					}

					return (l_str);

				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

			}
			public override string help()
			{
				return "Return Variable Value";
			}
		}
		/// <summary>
		/// Reset FIFO
		/// </summary>
		public class ResetFifo : ThreadCommand
		{
			public ResetFifo(ThreadCommandEnv x_command) : base(x_command, _CMD_RESET_FIFO) { }
			public override string Execute(string[] x_args)
			{
				try
				{
					//Reset FIFO
					for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart; l_i < m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd + 1; l_i++)
					{
						m_command.BMLib.ChainTdmFbw.TDMResetFifoOneBoard(l_i);
					}
					return "Reset Fifo";
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: Reset Fifo. \n";
			}
		}

		/// <summary>
		/// Set SYNC
		/// </summary>
		public class Sync : ThreadCommand
		{
			public Sync(ThreadCommandEnv x_command) : base(x_command, _CMD_SYNC) { }
			public override string Execute(string[] x_args)
			{
				try
				{
					//Prepare SYNC
					m_command.BMLib.ChainTdmFbw.ChainSetCheckSyncAndGtx(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart, m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd);

				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}

				return "Set SYNC on MCR " + (m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart /8).ToString() +"\n";
			}
			public override string help()
			{
				return " Usage: set SYNC Ext clk. \n";
			}
		}

		/// <summary>
		/// Prepare DAQ. Ready for taking data
		/// </summary>
		public class PrepareDAQCmd : ThreadCommand
		{
			public PrepareDAQCmd(ThreadCommandEnv x_command) : base(x_command, _CMD_PREPARE_DAQ) { }

			public override string Execute(string[] x_args)
			{
				string l_str = "";
				try
				{
					#region Parameters
					//Data and configuration file path
					string l_daqFile = m_command.BMLib.MCRSettings.DaqPath +  
						m_command.BMLib.MCRSettings.DaqType + 
						m_command.BMLib.MCRSettings.DaqFileName + //test_MCR
						(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart / 8).ToString() + "_Run" +//MCR number
						m_command.BMLib.MCRSettings.RunNumber+ "_" +
						string.Format("{0:yyyy_MM_dd_HH_mm_ss}", DateTime.Now) + ".daq";

					m_command.BMLib.MCRSettings.FullName = l_daqFile; //Save the daq file name in appsettings  

                    //Create dir if does not exist 
                    if (!Directory.Exists(m_command.BMLib.MCRSettings.DaqPath + m_command.BMLib.MCRSettings.DaqType))
                        Directory.CreateDirectory(m_command.BMLib.MCRSettings.DaqPath + m_command.BMLib.MCRSettings.DaqType);

                    //Configuration files
                    string[] l_cfgfilenames = m_command.BMLib.MCRSettings.ConfigurationFiles.ToArray();

                    #endregion

                    
                    //Check Is transferring data
                    if (m_command.BMLib.Link.IsTransferingData)
                    {
                        l_str = "MCR " + m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart / 8 + " is transferring data, cannot start a new DAQ.\n";
                        return l_str;
                    }

					l_str += "Prepare DAQ on MCR " + m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart / 8 + "\n";

					l_str += "Reset FIFO \n";
                    //Reset FIFO
                    for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart; l_i < m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd + 1; l_i++)
                    {
                        m_command.BMLib.ChainTdmFbw.TDMResetFifoOneBoard(l_i);
                    }
					l_str += "Check SYNC \n";
					try
					{
						//Only Check SYNC
						m_command.BMLib.ChainTdmFbw.ChainCheckSyncAndGtx(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart, m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd);
					}
					catch (Exception x_e)
					{
						m_command.Log.WriteLine("Error during check SYNC: " + x_e.ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
						return GenericCommand._ERR_RETURN + "Error during check SYNC: " + x_e.ToString();
					}
                    try
                    {
                        //Prepare DAQ
                        m_command.BMLib.ChainTdmFbw.TDMPrepareDaq(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart, m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd, l_cfgfilenames, 0);
                    }
                    catch (Exception x_e)
					{
						m_command.Log.WriteLine("Error during PrepareDAQ: "+ x_e.ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
						return GenericCommand._ERR_RETURN + "Error during PrepareDAQ: " + x_e.ToString();
					}

                        l_str = l_str + "Configuration files:\n";
					for (byte l_i =0; l_i< m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd- m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart +1; l_i ++)
					{
						l_str = l_str + l_cfgfilenames[l_i].ToString() + "\n";
					}
					l_str = l_str + "DAQ file: " + l_daqFile + "\n";
					
					l_str = l_str + "File size limit: " + m_command.BMLib.MCRSettings.FileSizeLimit + "\n";

					m_command.BMLib.Link.SetVariable("Board.UsbParam.FileLimit", m_command.BMLib.MCRSettings.FileSizeLimit);
					m_command.BMLib.Link.SetVariable("Board.UsbParam.Timeout", m_command.BMLib.MCRSettings.TimeLimit);

					l_str += "HV ON, \n";
					m_command.BMLib.ChainTdmFbw.ChainSetHV(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart, m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd, true);
					
					l_str += "Start DAQ.\n";
					m_command.BMLib.ChainTdmFbw.TDMStartDaq(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart, m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd, l_daqFile, 
															0, x_UDPreadout: m_command.BMLib.MCRSettings.Readout.UDPReadout);

					l_str += "Waiting for the trigger signal.\n";
					
				}
				catch (Exception x_e)
				{
					l_str = printException(x_e);
					m_command.Log.WriteLine("Error during PrepareDAQ: " + l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"Error during PrepareDAQ: " + $"[{l_str}]";
				}
				return l_str;
			}
			public override string help()
			{
				return " Usage: Prepare All MCRs for satrt of DAQ. \n";
			}
		}


		/// <summary>
		/// Stop DAQ. HV Off.
		/// </summary>
		public class StopDAQCmd : ThreadCommand
		{
			public StopDAQCmd(ThreadCommandEnv x_command) : base(x_command, _CMD_STOP_DAQ) { }

			public override string Execute(string[] x_args)
			{
				string l_str = "";
				try
				{
					const int l_timeout = 500;
					const int l_delay_ms = 10;

					l_str += "\nStop DAQ on MCR "+ m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart/8 + "\n";
                    l_str += "Is transfering data? "+ m_command.BMLib.Link.IsTransferingData.ToString() + "\n";
                    l_str += "DAQ file name: " + m_command.BMLib.MCRSettings.FullName + "\n";

                    m_command.BMLib.ChainTdmFbw.TDMStopDaq();

					int l_sleep = 0;
					while(m_command.BMLib.Link.IsTransferingData)
					{
						System.Threading.Thread.Sleep(l_delay_ms);
						l_sleep++;
						if (l_sleep > l_timeout)
							break;
					}

					l_str += "\nDAQ Total Time (dd.hh:mm:ss): " + m_command.BMLib.Link.ElapsedTime.ToString(@"dd\.hh\:mm\:ss") + "\n";
					l_str += "DAQ Total KBytes: " + m_command.BMLib.Link.XferBytes / 1000 + "\n";
					l_str += "DAQ Xfer rate (Bytes/ms): Avg=" + m_command.BMLib.Link.AvgXferRate + " / Max=" + m_command.BMLib.Link.MaxXferRate + "\n\n";

					if (l_sleep > l_timeout)
					{
						string l_error = $"Timeout error ({l_timeout*l_delay_ms} ms) while waiting for end of DAQ from user request";
						m_command.Log.WriteLine(l_error, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
						l_str += GenericCommand._ERR_RETURN + $"[{l_error}]\n";
						l_str += "Try to stop HV...\n";
					}
					else					
						l_str += "Stop HV\n";

					m_command.BMLib.ChainTdmFbw.ChainSetHV(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart, m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd, false);


					//Check error counters
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST) != 0)
					{
						l_str += ">>> Err in FSYNC_LOST counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST).ToString() + "\n";
						m_command.Log.WriteLine("FSYNC_LOST counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST) != 0)
					{
						l_str += ">>> Err in GTRIG_LOST counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST).ToString() + "\n";
						m_command.Log.WriteLine("GTRIG_LOST counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED) != 0)
					{
						l_str += ">>> Err in PLL_LOCKED counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED).ToString() + "\n";
						m_command.Log.WriteLine("PLL_LOCKED counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF) != 0)
					{
						l_str += ">>> Err in PARITY_EOF counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF).ToString() + "\n";
						m_command.Log.WriteLine("PARITY_EOF counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					l_str += "\n";
				}

				catch (Exception x_e)
				{
					l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
				return l_str;
			}
			public override string help()
			{
				return (" Usage: Stop.\n  DAQ Stop DAQ");
			}

		}

		/// <summary>
		/// Monitoring 
		/// </summary>
		public class Monitoring : ThreadCommand
		{
			public Monitoring(ThreadCommandEnv x_command) : base(x_command, _CMD_MONITORING) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = "";
					if (!m_command.BMLib.Link.ReadStatus())
					{
						return "Cannot access MCR " + m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart / 8 + " \n";
					}

					for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart; 
						l_i< m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd+1; l_i ++)
					{
						m_command.BMLib.Link.SetBoardId(l_i);
						l_str += "\nBoard ID: "+ l_i +"\n";
						l_str += "Slow Control Variables:\n" + m_command.BMLib.Link.GetHouseKeepingStrValues();
					}
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: OnlineMonitoring.\n Get Slow Control monitoring values";
			}
		}
		/// <summary>
		/// Monitoring One Channel
		/// </summary>
		public class MonitoringOneCh : ThreadCommand
		{
			public MonitoringOneCh(ThreadCommandEnv x_command) : base(x_command, _CMD_MONITORING_ONE_CH) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = "";
					bool l_BID_found = false;
					for (byte l_i = m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart;
						l_i < m_command.BMLib.MCRSettings.UsbBoardId.BoardIdEnd + 1; l_i++)
					{
						// x_args[1] = MCR#, x_args[2] = BID#, x_args[3] = HouseKeepingChannel
						
						if (l_i.ToString() == x_args[2])
						{
							l_BID_found = true;
							m_command.BMLib.Link.SetBoardId(l_i);
							Lib.LibBoardLink.HouseKeepingChannel l_channel=0;
							int l_ch = Convert.ToInt32(x_args[2]);
							if (x_args[3] == "0") l_channel = Lib.LibBoardLink.HouseKeepingChannel.TEMP_ASIC0;
							if (x_args[3] == "1") l_channel = Lib.LibBoardLink.HouseKeepingChannel.TEMP_ASIC1;
							if (x_args[3] == "2") l_channel = Lib.LibBoardLink.HouseKeepingChannel.TEMP_ASIC2;
							if (x_args[3] == "3") l_channel = Lib.LibBoardLink.HouseKeepingChannel.TEMP_FPGA;
							if (x_args[3] == "4") l_channel = Lib.LibBoardLink.HouseKeepingChannel.TEMP_BOARD;
							if (x_args[3] == "5") l_channel = Lib.LibBoardLink.HouseKeepingChannel.HR_BOARD;
							if (x_args[3] == "6") l_channel = Lib.LibBoardLink.HouseKeepingChannel.HV;
							if (x_args[3] == "7") l_channel = Lib.LibBoardLink.HouseKeepingChannel.NB_CHANNELS;

							l_str += m_command.BMLib.Link.GetHouseKeepingValue(l_channel).ToString();								
						}												
					}
					if (l_BID_found == false) {
						l_str += "BID was not found!! ";
					}
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return " Usage: Get HouseKeeping Values per channel.\n";
			}
		}


		/// <summary>
		/// Run status on demand
		/// </summary>
		public class RunStatusCmd : ThreadCommand
		{
			public RunStatusCmd(ThreadCommandEnv x_command) : base(x_command, _CMD_RUN_STATUS) { }

			public override string Execute(string[] x_args)
			{
				string l_str = "";
				try
				{
					l_str += "Run Status for MCR "+ m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart/8 + "\n";

					m_command.BMLib.Link.SetBoardId(m_command.BMLib.MCRSettings.UsbBoardId.BoardIdStart);
                    l_str += "Is transfering data: "+ m_command.BMLib.Link.IsTransferingData.ToString() + "\n";
                    l_str += "DAQ file name: " + m_command.BMLib.MCRSettings.FullName + "\n";
                    l_str += "DAQ Total Time (dd.hh:mm:ss): " + m_command.BMLib.Link.ElapsedTime.ToString(@"dd\.hh\:mm\:ss") + "\n";
					l_str += "DAQ Total KBytes: " + m_command.BMLib.Link.XferBytes / 1000 + "\n";
					l_str += "DAQ Xfer rate (Bytes/ms): Avg=" + m_command.BMLib.Link.AvgXferRate + " / Max=" + m_command.BMLib.Link.MaxXferRate + "\n";

					//Check error counters
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST) != 0)
					{
						l_str += ">>> Err in FSYNC_LOST counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST).ToString() + "\n";
						m_command.Log.WriteLine("FSYNC_LOST counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.FSYNC_LOST).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST) != 0)
					{ 
						l_str += ">>> Err in GTRIG_LOST counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST).ToString() + "\n";
						m_command.Log.WriteLine("GTRIG_LOST counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.GTRIG_LOST).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED) != 0) { 
						l_str += ">>> Err in PLL_LOCKED counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED).ToString() + "\n";
						m_command.Log.WriteLine("PLL_LOCKED counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PLL_LOCKED).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}
					if (m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF) != 0) { 
						l_str += ">>> Err in PARITY_EOF counter " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF).ToString() + "\n";
						m_command.Log.WriteLine("PARITY_EOF counter: " + m_command.BMLib.Link.GetErrorCounterValue(Lib.LibBoardLink.ErrorCounters.PARITY_EOF).ToString(), UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					}

					l_str += "\n";
				}
				catch (Exception x_e)
				{
					l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
				return l_str;
			}
			public override string help()
			{
				return "Usage: GetRunStatus.\n  Get Run status on demand";
			}
		}

		/// <summary>
		/// DataRate
		/// </summary>
		public class DataRate : ThreadCommand
		{
			public DataRate(ThreadCommandEnv x_command) : base(x_command, _CMD_DATA_RATE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = "";
					l_str += m_command.BMLib.Link.AvgXferRate;
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return "Usage: GetDataRate.\n";
			}
		}

		/// <summary>
		/// MaxRate
		/// </summary>
		public class MaxRate : ThreadCommand
		{
			public MaxRate(ThreadCommandEnv x_command) : base(x_command, _CMD_MAX_RATE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = m_command.BMLib.Link.MaxXferRate.ToString();
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return "Usage: Get Max Rate.\n";
			}
		}

		/// <summary>
		/// TimeElapsed
		/// </summary>
		public class TimeElapsed : ThreadCommand
		{
			public TimeElapsed (ThreadCommandEnv x_command) : base(x_command, _CMD_TIME) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = m_command.BMLib.Link.ElapsedTime.ToString();
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return "Usage: TimeElapsed.\n";
			}
		}
		/// <summary>
		/// DataFileName
		/// </summary>
		public class DataFileName : ThreadCommand
		{
			public DataFileName(ThreadCommandEnv x_command) : base(x_command, _CMD_FILE_NAME) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = m_command.BMLib.MCRSettings.FullName;
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return "Usage: Data File name.\n";
			}
		}
		/// <summary>
		/// FileSize
		/// </summary>
		public class FileSize : ThreadCommand
		{
			public FileSize(ThreadCommandEnv x_command) : base(x_command, _CMD_FILE_SIZE) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = (m_command.BMLib.Link.XferBytes/1000).ToString() ; 
					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return "Usage: File Size.\n";
			}
		}
		/// <summary>
		/// Set Run Properties
		/// </summary>
		public class SetRunPropCmd : ThreadCommand
		{
			public SetRunPropCmd(ThreadCommandEnv x_command) : base(x_command, _CMD_SET_RUN_PROP) { }

			public override string Execute(string[] x_args)
			{
				try
				{
					string l_str = "RunProperties: \nRun Number: " + x_args[1] + "\nRun Type: " + x_args[2] +"\n";
					m_command.BMLib.MCRSettings.RunNumber = x_args[1];
					m_command.BMLib.MCRSettings.DaqType = x_args[2];

					return l_str;
				}
				catch (Exception x_e)
				{
					string l_str = printException(x_e);
					m_command.Log.WriteLine(l_str, UnigeFrontEnd.Tools.Logger.Severity.CRITICAL);
					return GenericCommand._ERR_RETURN + $"[{l_str}]";
				}
			}
			public override string help()
			{
				return "Usage: Set Run Properties";
			}
		}
		
	}	
}
