library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_arith.ALL;
use IEEE.STD_LOGIC_signed.ALL;

-- This source code describes a single channel FIR Filter implemented on the Systolic structure
-- The N-tap FIR implementation requires N DSP blocks

entity Systolic_flat is
generic ( 
   DIN_width : integer range 8 to 18 := 18;
   Coefs_width : integer range 8 to 18 := 18;
   Output_width : integer range 8 to 50 := 18 
   );
port ( 
   CLK   : in  STD_LOGIC;
   DIN   : in  STD_LOGIC_VECTOR (DIN_width-1 downto 0);
   DOUT : out  STD_LOGIC_VECTOR (Output_width-1 downto 0)
   );
end Systolic_flat;

architecture Behavioral of Systolic_flat is

signal FIR_OUT_INT : std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0);

signal FIR_OUT : std_logic_vector(Output_width-1 downto 0);
signal FIR_OUT_R : std_logic_vector(Output_width-1 downto 0);

-- If the coefficient value are in hexadecimal, they must be declared on a multiple of 4 bits.
-- The additional bits to get a 4-bit boundary are simply discarded
type COEFS_TYPE is array (natural range <>) of std_logic_vector((((Coefs_width+3)/4)*4)-1 downto 0);
constant COEFS : COEFS_TYPE := (
-- First                             Central                              Last
-- Coefs(0), Coefs(1), Coefs(2), Coefs(3), Coefs(4), Coefs(5), Coefs(6), Coefs(7), 
   x"01111", x"02222", x"03333", x"04444", x"05555", x"06666", x"07777", x"08888"
   );
--signal S_COEFS : COEFS_TYPE := (
--   x"01111", x"02222", x"03333", x"04444", x"05555", x"06666", x"07777"
--   ); 

constant ROUND_VAL : std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0) := --(others => '0');
(DIN_width + Coefs_width-1 - Output_width => '1', others => '0'); 

signal S_ROUND_VAL : std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0);

type DINR_TYPE is array(0 to COEFS'high) of std_logic_vector(DIN_width-1 downto 0);
signal DINR1 : DINR_TYPE;

type MULT_TYPE is array(0 to COEFS'high) of std_logic_vector(DIN_width + Coefs_width-1 downto 0);
signal MULT : MULT_TYPE;
attribute NX_USE : string;
attribute NX_USE of MULT : signal is "NX_DSP";

type ADD_TYPE is array(0 to COEFS'high) of std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0);
signal ADD : ADD_TYPE;
type DATA_CHAIN_TYPE is array(0 to COEFS'high) of std_logic_vector(DIN_width-1 downto 0);
signal DATA_CHAIN : DATA_CHAIN_TYPE;

begin

--S_COEFS <= COEFS;
S_ROUND_VAL <= ROUND_VAL;


process(CLK)  begin
   if rising_edge(CLK)  then
      for I in 0 to COEFS'high  loop
         if I = 0  then
            DATA_CHAIN(I) <= DIN;  -- Single pipe level for the first tap
            MULT(I) <= DATA_CHAIN(I) * COEFS(I)(Coefs_width-1 downto 0);
            ADD(I) <= MULT(I) + S_ROUND_VAL;
         else
            DINR1(I) <= DATA_CHAIN(I-1);
            DATA_CHAIN(I) <= DINR1(I); -- Dual pipe level for all other taps
            MULT(I) <= DATA_CHAIN(I) * COEFS(I)(Coefs_width-1 downto 0);
            ADD(I) <= MULT(I) + ADD(I-1);
         end if;
      end loop;
   end if;
end process;

FIR_OUT_INT <= ADD(COEFS'length - 1);

-- Discarding the unsignificant MSBs and LSB truncation
DOUT <= FIR_OUT_INT(DIN_width + Coefs_width-1 downto DIN_width + Coefs_width - Output_width);  

end Behavioral;

