library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_arith.ALL;
use IEEE.STD_LOGIC_signed.ALL;

-- This source code describes a single channel FIR Filter implemented on the Systolic structure
-- The N-tap FIR implementation requires N DSP blocks

entity Systolic_hier is
generic ( 
   DIN_width : integer range 8 to 18 := 18;
   Coefs_width : integer range 8 to 18 := 18;
   Output_width : integer range 8 to 50 := 18 
   );
port ( 
   CLK   : in  STD_LOGIC;
   DIN   : in  STD_LOGIC_VECTOR (DIN_width-1 downto 0);
   DOUT : out  STD_LOGIC_VECTOR (Output_width-1 downto 0)
   );
end Systolic_hier;

architecture Behavioral of Systolic_hier is

component Systolic_cell 
generic ( 
   DIN_width : integer range 8 to 18 := 18;
   Coefs_width : integer range 8 to 18 := 18;
   Output_width : integer range 8 to 50 := 18 
   );
port ( 
   CLK   : in  STD_LOGIC;
   COEF  : in std_logic_vector(Coefs_width-1 downto 0);
   DATA_CHAIN_IN   : in  std_logic_vector(DIN_width-1 downto 0);
   DATA_CHAIN_OUT  : out std_logic_vector(DIN_width-1 downto 0);
   ADD_CHAIN_IN    : in std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0);
   ADD_CHAIN_OUT   : out std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0)
   );
end component;

-- If the coefficient value are in hexadecimal, they must be declared on a multiple of 4 bits.
-- The additional bits to get a 4-bit boundary are simply discarded
type COEFS_TYPE is array (natural range <>) of std_logic_vector((((Coefs_width+3)/4)*4)-1 downto 0);
constant COEFS : COEFS_TYPE := (
-- First                             Central                              Last
-- Coefs(0), Coefs(1), Coefs(2), Coefs(3), Coefs(4), Coefs(5), Coefs(6), Coefs(7), 
   x"01111", x"02222", x"03333", x"04444", x"05555", x"06666", x"07777", x"08888"
   );

constant ROUND_VAL : std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0) := 
                                 (DIN_width + Coefs_width-1 - Output_width => '1', others => '0'); 

signal S_ROUND_VAL : std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0);

type DATA_CHAIN_TYPE is array(0 to COEFS'high) of std_logic_vector(DIN_width-1 downto 0);
signal DATA_CHAIN : DATA_CHAIN_TYPE;

type ADD_CHAIN_TYPE is array(0 to COEFS'high) of std_logic_vector(DIN_width + Coefs_width-1 + 3 downto 0);
signal ADD_CHAIN : ADD_CHAIN_TYPE;

begin

S_ROUND_VAL <= ROUND_VAL;

GEN_CELLS : for I in 0 to COEFS'high  generate
   G0 : if I = 0  generate
   CELL0 : Systolic_cell 
      generic map( 
         DIN_width    => DIN_width,
         Coefs_width  => Coefs_width,
         Output_width => Output_width 
         )
      port map( 
         CLK             => CLK,
         COEF            => COEFS(0)(Coefs_width-1 downto 0),
         DATA_CHAIN_IN   => DIN,
         DATA_CHAIN_OUT  => DATA_CHAIN(0),
         ADD_CHAIN_IN    => S_ROUND_VAL,
         ADD_CHAIN_OUT   => ADD_CHAIN(0)
      );
   end generate;
   
   Gx : if I /= 0  generate
   CELLx : Systolic_cell 
      generic map( 
         DIN_width    => DIN_width,
         Coefs_width  => Coefs_width,
         Output_width => Output_width 
         )
      port map( 
         CLK             => CLK,
         COEF            => COEFS(I)(Coefs_width-1 downto 0),
         DATA_CHAIN_IN   => DATA_CHAIN(I-1),
         DATA_CHAIN_OUT  => DATA_CHAIN(I),
         ADD_CHAIN_IN    => ADD_CHAIN(I-1),
         ADD_CHAIN_OUT   => ADD_CHAIN(I)
      );
   end generate;
end generate;   

-- Discarding the unsignificant MSBs and LSB truncation
DOUT <= ADD_CHAIN(COEFS'high)(DIN_width + Coefs_width-1 downto DIN_width + Coefs_width - Output_width);  

end Behavioral;

