library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_arith.ALL;
use IEEE.STD_LOGIC_signed.ALL;

-- This source code describes a symmetrical FIR Filter implemented on the Transpose symmetric structure
-- The N-tap FIR implementation requires N/2 DSP blocks

entity transpose_sym_flat is
port ( 
   CLK   : in  STD_LOGIC;
   DIN   : in  STD_LOGIC_VECTOR (11 downto 0);
   DOUT  : out STD_LOGIC_VECTOR (15 downto 0)
   );
end transpose_sym_flat;

architecture Behavioral of transpose_sym_flat is

signal DIN_IO : std_logic_vector(11 downto 0);  -- I/O input register
signal DIN_TILE : std_logic_vector(11 downto 0);  -- Tile register on the input path
attribute syn_preserve : boolean;
attribute syn_preserve of DIN_TILE : signal is true;

-- This 20-Tap symmetric FIR filter requires 10 coefficients 
type COEFS_TYPE is array (0 to 9) of integer range -2048 to 2047;
constant COEFS : COEFS_TYPE := (
-- First                         Central  
-- Coefs(0), Coefs(1), Coefs(2), Coefs(3)
   -37, -40, 34, 127, 75, -155, -286, 48, 823, 1497
   );

type S_COEFS_TYPE is array(0 to 9) of std_logic_vector(11 downto 0);
signal S_COEFS : S_COEFS_TYPE;

constant ROUND_VAL : std_logic_vector(29 downto 0) := (7 => '1',  others => '0');
signal S_ROUND_VAL : std_logic_vector(29 downto 0);

type DINR_TYPE is array(0 to 9) of std_logic_vector(11 downto 0);
signal DINR, DINR1, DIN_CHAIN : DINR_TYPE;

type PREAD_TYPE is array(0 to 9) of std_logic_vector(12 downto 0);
signal PREAD : PREAD_TYPE;

constant ZERO : std_logic_vector(12 downto 0) := (others => '0');

type MULT_TYPE is array(0 to 9) of std_logic_vector(24 downto 0);
signal MULT : MULT_TYPE;

type ADD_TYPE is array(0 to 9) of std_logic_vector(29 downto 0);
signal ADD : ADD_TYPE;

attribute NX_USE :string;
attribute NX_USE of MULT: signal is "NX_DSP";

begin

GENSCOEFS : for I in 0 to 9  generate
    S_COEFS(I)     <= conv_std_logic_vector(COEFS(I), 12);
end generate;

    S_ROUND_VAL <= ROUND_VAL;

process(CLK)  begin
   if rising_edge(CLK)  then
      DIN_IO <= DIN;
      DIN_TILE <= DIN_IO;
   end if;
end process;      

process(CLK)  begin
   if rising_edge(CLK)  then
      for I in 0 to 9 loop
         DINR(I) <= DIN_TILE;
         if I = 0  then
            DINR1(I) <= DIN_TILE;
            DIN_CHAIN(I) <= DINR1(I);
            PREAD(I) <= ZERO + DINR(I) + DIN_CHAIN(I); 
            MULT(I) <= PREAD(I) * S_COEFS(9 - I);
            ADD(I) <= MULT(I) + S_ROUND_VAL;
         else
            DINR1(I) <= DIN_CHAIN(I-1);
            DIN_CHAIN(I) <= DINR1(I);
            PREAD(I) <= ZERO + DINR(I) + DIN_CHAIN(I); 
            MULT(I) <= PREAD(I) * S_COEFS(9 - I);
            ADD(I) <= MULT(I) + ADD(I-1);
         end if;
      end loop;
   end if;
end process;


process(CLK)  begin
   if rising_edge(CLK)  then
      DOUT <= ADD(9)(23 downto 8);	    
   end if;
end process;      

end Behavioral;

