library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_arith.ALL;
use IEEE.STD_LOGIC_signed.ALL;

entity TRANSP_SYM_HIER is
generic (
   DIN_WIDTH : integer range 8 to 18 := 18;
   COEF_WIDTH : integer range 8 to 24 := 18;
   DOUT_WIDTH : integer range 8 to 50 := 18
   );
port ( 
   CLK : in  STD_LOGIC;
   DIN : in std_logic_vector (DIN_WIDTH-1 downto 0);
   DOUT : out std_logic_vector (DOUT_WIDTH-1 downto 0)
   );
end TRANSP_SYM_HIER;

architecture Behavioral of TRANSP_SYM_HIER is

component PREAD_MULT_ADD_CHAIN 
generic (
   DIN_WIDTH : integer range 8 to 18 := 18;
   COEF_WIDTH : integer range 8 to 24 := 18
   );
port ( 
   CLK : in  STD_LOGIC;
   DINL : in std_logic_vector(DIN_WIDTH-1 downto 0);
   DINR : in std_logic_vector(DIN_WIDTH-1 downto 0);
   D_CHAIN : out std_logic_vector(DIN_WIDTH-1 downto 0);
   COEF : in std_logic_vector(COEF_WIDTH-1 downto 0);
   ADD_CHAIN_IN : in std_logic_vector(DIN_WIDTH + COEF_WIDTH + 6 downto 0);
   ADD_CHAIN_OUT : out std_logic_vector(DIN_WIDTH + COEF_WIDTH + 6 downto 0)
   );
end component;

-- Number of taps must be even
-- Only half of coefs values are declared, starting by the most right coef, and ending by the center coefs  
type COEFS_TYPE is array (natural range <>) of std_logic_vector((((COEF_WIDTH + 3)/4)*4)-1 downto 0);
constant COEFS : COEFS_TYPE := (
   x"01111", x"02222", x"03333", x"04444", x"05556"
   );
   
type COEF_TYPE is array (COEFS'range) of std_logic_vector(COEF_WIDTH-1 downto 0);
signal COEF : COEF_TYPE;
   
type D_CHAIN_TYPE is array(COEFS'length-1 downto 0) of std_logic_vector(DIN_WIDTH -1 downto 0);
signal D_CHAIN : D_CHAIN_TYPE;

type ADD_CHAIN_TYPE is array(COEFS'length-1 downto 0) of std_logic_vector(DIN_WIDTH + COEF_WIDTH + 6 downto 0);
signal ADD_CHAIN : ADD_CHAIN_TYPE;

signal ROUND_VAL : std_logic_vector(DIN_WIDTH + COEF_WIDTH + 6 downto 0);

begin

ROUND_VAL <= (DIN_WIDTH + COEF_WIDTH - DOUT_WIDTH - 1 => '1', others => '0');
GENCO : for I in COEFS'high downto 0  generate
   COEF(I) <= COEFS(I)(COEF_WIDTH-1 downto 0);
end generate;

GCELL : for I in 0 to COEFS'length-1  generate
   GFIRST : if I = 0  generate
      CELL_0 : PREAD_MULT_ADD_CHAIN 
         generic map(
            DIN_WIDTH  => 18,
            COEF_WIDTH => 18
            )
         port map( 
            CLK           => CLK,
            DINL          => DIN,
            DINR          => DIN,
            D_CHAIN       => D_CHAIN(0),
            COEF          => COEF(COEFS'high),
            ADD_CHAIN_IN  => ROUND_VAL,
            ADD_CHAIN_OUT => ADD_CHAIN(0)
            );
   end generate;
   GOTH : if I /= 0  generate
      CELL_X : PREAD_MULT_ADD_CHAIN 
         generic map(
            DIN_WIDTH  => 18,
            COEF_WIDTH => 18
            )
         port map( 
            CLK           => CLK,
            DINL          => D_CHAIN(I-1),
            DINR          => DIN,
            D_CHAIN       => D_CHAIN(I),
            COEF          => COEF(COEFS'high-I),
            ADD_CHAIN_IN  => ADD_CHAIN(I-1),
            ADD_CHAIN_OUT => ADD_CHAIN(I)
            );
   end generate;
end generate;

DOUT <= ADD_CHAIN(COEFS'high)(COEF_WIDTH + DIN_WIDTH - 1 downto COEF_WIDTH + DIN_WIDTH - DOUT_WIDTH);

end Behavioral;

