library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity NX_SYSTOLIC_CELL is
generic(
   COEF_WIDTH : integer range 10 to 24 := 24;
   DATA_WIDTH : integer range 10 to 18 := 18;
   Position : string := "First"; -- := "Last";  --"First"  --
   ENABLE_SATURATION_CST : std_logic := '0';
   SATURATION_RANK_CST : std_logic_vector(5 downto 0) := "100000";
   MUX_B_CST : std_logic := '0';
   PR_B_CASCADE_MUX_CST : std_logic_vector(1 downto 0) := "01"; -- "01" for one pipe level, "10" for 2 levels
   MUX_X_CST : std_logic_vector(1 downto 0) := "00"  -- 00 for C input (sign extended), "01" for CZI input
   );
Port ( 
   CLK : in  STD_LOGIC;
   SRST : in  STD_LOGIC;
   ENA : in  STD_LOGIC;
   CASC_IN_B : in std_logic_vector(17 downto 0);   -- DIN filter or CASC_B from previous DSP
   CASC_IN_Z : in std_logic_vector(55 downto 0);   -- CASC_Z
   COEF : in  STD_LOGIC_VECTOR (COEF_WIDTH-1 downto 0);
   OVF : out std_logic;
   CASC_OUT_B : out std_logic_vector(17 downto 0);
   CASC_OUT_Z : out  STD_LOGIC_VECTOR (55 downto 0)
   );
end NX_SYSTOLIC_CELL;

architecture Behavioral of NX_SYSTOLIC_CELL is

component DSP_SPLIT_GENERICS is
generic (
-------------------------------------------------------------------------
-- Generic declaration to define the "raw_config0" (cfg_mode). Defines :
------------------------------------------------------------------------- 
   SIGNED_MODE : std_logic              := '1';
   PRE_ADDER_OP : std_logic             := '0';  -- '0' = Additon, '1' = Subraction
   MUX_A : std_logic                    := '0';  -- '0' = A input, '1' = CAI input
   MUX_B : std_logic                    := '0';  -- '0' = B input, '1' = CBI input
   MUX_P : std_logic                    := '0';  -- '0' for PRE_ADDER, '0' for B input
   MUX_X : std_logic_vector(1 downto 0) := "01"; -- Select X operand   "00" = C, "01" = CZI, "11" = SHFT(CZI) & C(11:0), "10" Select Z feedback 
   MUX_Y : std_logic                    := '0';  -- Select MULT output
   MUX_CI : std_logic                   := '0';  -- Select fabric input (not cascade)
   MUX_Z : std_logic                    := '0';  -- Select ALU output (not ALU operand input input (instead of CAI)
   
   Z_FEEDBACK_SHL12 : std_logic         := '0';  -- '0' for No shift, '1' for 12-bit left shift
   ENABLE_SATURATION : std_logic        := '0';  -- '0' for Disable, '1' for Enable
   SATURATION_RANK : std_logic_vector(5 downto 0) := "110110"; -- Weight of useful MSB on Z and CZO result (to define saturation and overflow) 
   
   ALU_DYNAMIC_OP : std_logic           := '0';  -- '0' for Static, '1' for Dynamic (D6 ... D1 is not used for dynamic operation)
   CO_SEL : std_logic                   := '0';  -- '0' for C0 = ALU(36), '1' for CO = ALU(48)   

-------------------------------------------------------------------------
-- Generic declaration to define the "raw_config1" (cfg_pipe_mux)
-------------------------------------------------------------------------
   PR_A_MUX : std_logic_vector(1 downto 0) := "01";         -- Number of pipe reg levels on A input
   PR_A_CASCADE_MUX : std_logic_vector(1 downto 0) := "00"; -- Number of pipe reg levels for CAO output
   PR_B_MUX : std_logic_vector(1 downto 0) := "01";         -- Number of pipe reg levels on B input
   PR_B_CASCADE_MUX : std_logic_vector(1 downto 0) := "00"; -- Number of pipe reg levels for CAO output
   
   PR_C_MUX : std_logic    := '0';       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_D_MUX : std_logic    := '0';       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_CI_MUX : std_logic   := '1';       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_P_MUX : std_logic    := '1';       -- '0' for No pipe reg, '1' for 1 pipe reg (Pre-adder)
   PR_X_MUX : std_logic    := '0';       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_Y_MUX : std_logic    := '1';       -- '0' for No pipe reg, '1' for 1 pipe reg
   
   PR_MULT_MUX : std_logic := '1';       -- No pipe reg  -- Register inside MULT
   PR_ALU_MUX : std_logic  := '0';       -- No pipe reg  -- Register inside ALU
   PR_Z_MUX : std_logic    := '1';       -- Registered output
   
   PR_CO_MUX : std_logic   := '0';       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_OV_MUX : std_logic   := '1';       -- '0' for No pipe reg, '1' for 1 pipe reg

-------------------------------------------------------------------------
-- Generic declaration to define the "raw_config2" (cfg_pipe_rst)
-------------------------------------------------------------------------
   ENABLE_PR_A_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_B_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_C_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_D_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_CI_RST : std_logic   := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_P_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_X_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_Y_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_MULT_RST : std_logic := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_ALU_RST : std_logic  := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_Z_RST : std_logic    := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_CO_RST : std_logic   := '1';   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_OV_RST : std_logic   := '1';   -- '0' for Disable, '1' for Enable 

-------------------------------------------------------------------------
-- Constants declaration to define the "cfg_pipe_rst" -- raw_config3(6 downto 0)
-------------------------------------------------------------------------
   ALU_OP : std_logic_vector(5 downto 0) := "000000";  -- Addition = "000000", Subtract = "001010"
   ALU_MUX : std_logic                   := '0'  -- '0' for Don't swap ALU operands, '1' for ALU Swap operands
    );
port(
   CK : IN  std_logic;
   R : IN  std_logic;
   RZ : IN  std_logic;
   WE : IN  std_logic;
   
   CI : IN  std_logic;  -- cy_i
   A : IN  std_logic_vector(23 downto 0);
   B : IN  std_logic_vector(17 downto 0);
   C : IN  std_logic_vector(35 downto 0);
   D : IN  std_logic_vector(17 downto 0);
   CAI : IN  std_logic_vector(17 downto 0);
   CBI : IN  std_logic_vector(17 downto 0);
   CZI : IN  std_logic_vector(55 downto 0);
   CCI : IN  std_logic;  -- ccy_i

   Z : out  std_logic_vector(55 downto 0);
   CO : OUT  std_logic;  -- cy_o
   CO36 : OUT  std_logic;  -- cy36_o
   CO48 : OUT  std_logic;  -- cy48_o
   OVF : OUT  std_logic;
   CAO : OUT  std_logic_vector(17 downto 0);
   CBO : OUT  std_logic_vector(17 downto 0);
   CZO : OUT  std_logic_vector(55 downto 0);
   CCO : OUT  std_logic  -- ccy_o
  );
end component;

signal CZI, Z, CZO :  std_logic_vector(55 downto 0);

signal GND : std_logic;
signal ZERO : std_logic_vector(55 downto 0);

signal A_INPUT : std_logic_vector(23 downto 0);
signal B_INPUT : std_logic_vector(17 downto 0);

signal C : std_logic_vector(35 downto 0);

signal B, CBI : std_logic_vector(17 downto 0);

begin

GND <= '0';
ZERO <= (others => '0');

GEN_A_INP_COEF_24B : if COEF_WIDTH < 24  generate
   SIGN_EXT_COEF : for I in 23 downto COEF_WIDTH  generate
      A_INPUT(I) <= COEF(COEF_WIDTH-1);
   end generate;
end generate;
A_INPUT(COEF_WIDTH-1 downto 0) <= COEF;

C <= CASC_IN_Z(35 downto 0)  when Position = "First"  else ZERO(35 downto 0);
CZI  <= CASC_IN_Z  when Position /= "First"  else ZERO;
CBI  <= CASC_IN_B  when Position /= "First"  else ZERO(17 downto 0);
B    <= CASC_IN_B  when Position = "First"   else ZERO(17 downto 0);

CASC_OUT_Z <= Z  when Position = "Last"  else CZO;

DSP_CELL : DSP_SPLIT_GENERICS 
generic map(
-------------------------------------------------------------------------
-- Generic declaration to define the "raw_config0" (cfg_mode). Defines :
------------------------------------------------------------------------- 
   SIGNED_MODE  => '1',
   PRE_ADDER_OP => '0',       -- '0' = Additon, '1' = Subraction
   MUX_A        => '0',       -- '0' = A input, '1' = CAI input
   MUX_B        => MUX_B_CST,  -- '0' = B input, '1' = CBI input
   MUX_P        => '1',       -- '0' for PRE_ADDER, '1' for B input
   MUX_X        => MUX_X_CST, -- "01",  -- "00" = C, "01" = CZI, "11" = SHFT(CZI) & C(11:0), "10" Select Z feedback 
   MUX_Y        => '0',       -- '0' Select MULT output, '1' select B & A
   MUX_CI       => '0',       -- '0' = fabric input, '1' = not cascade input
   MUX_Z        => '0',       -- '0' = Select ALU output, '1' = ALU operand input
   
   Z_FEEDBACK_SHL12  => '0',  -- '0' for No shift, '1' for 12-bit left shift
   ENABLE_SATURATION => ENABLE_SATURATION_CST, -- '0' for Disable, '1' for Enable
   SATURATION_RANK   => SATURATION_RANK_CST, --"110000",   -- Weight of useful MSB on Z and CZO result (to define saturation and overflow) 
   
   ALU_DYNAMIC_OP => '0',  -- '0' for Static, '1' for Dynamic (D6 ... D1 is not used for dynamic operation)
   CO_SEL         => '0',          -- '0' for C0 = ALU(36), '1' for CO = ALU(48)   

-------------------------------------------------------------------------
-- Generic declaration to define the "raw_config1" (cfg_pipe_mux)
-------------------------------------------------------------------------
   PR_A_MUX         => "01",         -- Number of pipe reg levels on A input
   PR_A_CASCADE_MUX => "00",         -- Number of pipe reg levels for CAO output
   PR_B_MUX         => "01",         -- Number of pipe reg levels on B input
   PR_B_CASCADE_MUX => "10",         -- Number of pipe reg levels for CAO output
   
   PR_C_MUX         => '0',       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_D_MUX         => '0',       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_CI_MUX        => '0',      -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_P_MUX         => '0',       -- '0' for No pipe reg, '1' for 1 pipe reg (Pre-adder)
   PR_X_MUX         => '0',       -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_Y_MUX         => '1',       -- '0' for No pipe reg, '1' for 1 pipe reg
   
   PR_MULT_MUX      => '1',    -- Register inside MULT
   PR_ALU_MUX       => '0',     -- Register inside ALU
   PR_Z_MUX         => '1',       -- Registered Z output
   
   PR_CO_MUX        => '0',      -- '0' for No pipe reg, '1' for 1 pipe reg
   PR_OV_MUX        => '1',      -- '0' for No pipe reg, '1' for 1 pipe reg

-------------------------------------------------------------------------
-- Generic declaration to define the "raw_config2" (cfg_pipe_rst)
-------------------------------------------------------------------------
   ENABLE_PR_A_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_B_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_C_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_D_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_CI_RST => '1',  -- '0' for Disable, '1' for Enable 
   ENABLE_PR_P_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_X_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_Y_RST  => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_MULT_RST => '1',   -- '0' for Disable, '1' for Enable 
   ENABLE_PR_ALU_RST => '1',     -- '0' for Disable, '1' for Enable 
   ENABLE_PR_Z_RST  => '1',        -- '0' for Disable, '1' for Enable 
   ENABLE_PR_CO_RST => '1',       -- '0' for Disable, '1' for Enable 
   ENABLE_PR_OV_RST => '1',       -- '0' for Disable, '1' for Enable 

-------------------------------------------------------------------------
-- Constants declaration to define the "cfg_pipe_rst" -- raw_config3(6 downto 0)
-------------------------------------------------------------------------
   ALU_OP   => "000000",  -- Addition = "000000", Subtract = "001010"
   ALU_MUX  => '0'  -- '0' for Don't swap ALU operands, '1' for ALU Swap operands
    )
port map(
   CK    => CLK,
   R     => SRST,
   RZ    => SRST,
   WE    => ENA,
   
   CI    => GND,
   A     => A_INPUT,
   B     => B,
   C     => C,
   D     => ZERO(17 downto 0),
   CAI   => ZERO(17 downto 0),
   CBI   => CBI,
   CZI   => CZI,
   CCI   => GND,
   
   CAO   => open,
   CBO   => CASC_OUT_B,
   
   CO    => open,
   CCO   => open,
   CO36  => open,
   CO48  => open,
   OVF   => OVF,

   CZO   => CZO,
   Z     => Z
  );

end Behavioral;

