library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_arith.ALL;
use IEEE.STD_LOGIC_signed.ALL;

package NX_SYSTOLIC_FIR_pack is

-- N_TAPS (number_of_taps) can be any length 
-- from 3 to 112 (number of available DSP blocks in NG_MEDIUM)
-- 3 to 56 taps for very high performance (> 300 MHz)
--
-- Coefficients can be assigned with :
--    - Input signal (array of std_logic_vector)
--    - Hexadecimal constants (for not multiple of 4 bits the remaining MSBs are ignored)
--    - Integer constants

constant N_TAPS : integer range 3 to 56          := 14;    -- Number of taps

constant DATA_WIDTH : integer range 10 to 18     := 16;   -- Number of data bits
constant COEF_WIDTH : integer range 10 to 24     := 14;   -- Number of coefficient bits

-- For variable coefficent set, you must use signal (instead of constant) for COEFS
constant COEFS_ARE_INPUT_SIGNALS : boolean       := false; -- False if COEFS are constant

constant PKG_COEFS_VALUES_ARE_HEXADECIMAL : boolean := false; -- true if values in Hexa, false for decimal integer            
constant TO_BE_TRUNCATED : integer range 1 to 36 := 14;      -- LSBs to be truncated at the FIR output

constant SATURATION_RANK_CST : std_logic_vector(6 downto 0):= conv_std_logic_vector(DATA_WIDTH + TO_BE_TRUNCATED-1 , 7); 

-----------------------------------------
-- Don't modify those next two lines
-----------------------------------------
type COEFS_PKG_INTEGER_CONSTANT_TYPE is array(0 to N_TAPS-1) of integer range -(2**COEF_WIDTH-1) to (2**COEF_WIDTH-1)-1;
type COEFS_PKG_HEXA_CONSTANT_TYPE is array(0 to N_TAPS-1) of std_logic_vector((((COEF_WIDTH+3)/4) * 4)-1 downto 0);
-----------------------------------------

-----------------------------------------
-- Enter here your coefficients DECIMAL integer values 
-- Or comment the following constant value assignment
constant COEFS_PKG_INTEGER_CONSTANT : COEFS_PKG_INTEGER_CONSTANT_TYPE := (
--   others => 0
-- 1, -1, 1, -1, 1, -2, -146, -596, -644, 1235, 4745, 6637, 4745, 1235, -644, -596, -146, -2, 1, -1, 1, -1, 1 
-2, -146, -596, -644, 535, 3745, 5037, 5037, 3745, 535, -644, -596, -146, -2 
   );
   
-----------------------------------------
-- Enter here your coefficients HEXAdecimal values
-- Or comment the following constant declaration
constant COEFS_PKG_HEXA_CONSTANT : COEFS_PKG_HEXA_CONSTANT_TYPE := (others => (others => '0'));
--x"eeee", -- x"3ffe",
--x"3f6e",
--x"3dac",
--x"3d7c",
--x"04d3",
--x"1289",
--x"19ed",
--x"1289",
--x"04d3",
--x"3d7c",
--x"3dac",
--x"3f6e",
--x"3ffe"
--   );   
   
-----------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------

-----------------------------------------
-- Don't modify the following lines
-----------------------------------------
type COEFS_TYPE is array(0 to N_TAPS-1) of std_logic_vector(COEF_WIDTH-1 downto 0);
constant ROUND_VAL : std_logic_vector(35 downto 0) := (TO_BE_TRUNCATED-1 => '1', others => '0');   
constant COEFS_ARE_PKG_CONSTANT : boolean := not(COEFS_ARE_INPUT_SIGNALS); -- False if COEFS are constant
   
end NX_SYSTOLIC_FIR_pack;


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_arith.ALL;
use IEEE.STD_LOGIC_signed.ALL;

library work;
use work.NX_SYSTOLIC_FIR_pack.all;

entity NX_SYSTOLIC_FIR is
generic (
   DATA_WIDTH : integer range 8 to 18 := DATA_WIDTH;
   COEF_WIDTH : integer range 8 to 24 := COEF_WIDTH;
   N_TAPS : integer := N_TAPS
   );
Port ( 
   CLK : in  STD_LOGIC;
   SRST : in  STD_LOGIC;
   ENA : in  STD_LOGIC;
   DIN : in  STD_LOGIC_VECTOR (DATA_WIDTH-1 downto 0);
   COEFS_IN : in  COEFS_TYPE;
   OVF : out std_logic;
   FIR_OUT : out  STD_LOGIC_VECTOR (55 downto 0)
   );
end NX_SYSTOLIC_FIR;

architecture Behavioral of NX_SYSTOLIC_FIR is

component NX_SYSTOLIC_CELL 
generic(
   COEF_WIDTH : integer range 10 to 24 := 24;
   DATA_WIDTH : integer range 10 to 18 := 18;
   Position : string := "First"; -- := "Last";  --"First"  --
   ENABLE_SATURATION_CST : std_logic := '0';
   SATURATION_RANK_CST : std_logic_vector(5 downto 0) := "100000";
   MUX_B_CST : std_logic := '0';
   PR_B_CASCADE_MUX_CST : std_logic_vector(1 downto 0) := "01"; -- "01" for one pipe level, "10" for 2 levels
   MUX_X_CST : std_logic_vector(1 downto 0) := "00"  -- 00 for C input (sign extended), "01" for CZI input
   );
Port ( 
   CLK : in  STD_LOGIC;
   SRST : in  STD_LOGIC;
   ENA : in  STD_LOGIC;
   CASC_IN_B : in std_logic_vector(17 downto 0);
   CASC_IN_Z : in std_logic_vector(55 downto 0);
   COEF : in  STD_LOGIC_VECTOR (COEF_WIDTH-1 downto 0);
   OVF : out std_logic;
   CASC_OUT_B : out std_logic_vector(17 downto 0);
   CASC_OUT_Z : out  STD_LOGIC_VECTOR (55 downto 0)
   );
end component;

signal ROUND_VAL_EXT : std_logic_vector(55 downto 0);

signal COEFS_SRC : COEFS_TYPE;

signal DIN_EXT : std_logic_vector(17 downto 0);

type CASC_B_TYPE is array(0 to N_TAPS-1) of std_logic_vector(17 downto 0);
signal CASC_B : CASC_B_TYPE;
type CASC_Z_TYPE is array(0 to N_TAPS-1) of std_logic_vector(55 downto 0);
signal CASC_Z : CASC_Z_TYPE;

begin

ROUND_VAL_EXT(55 downto 36) <= (others => '0');
ROUND_VAL_EXT(35 downto 0) <= ROUND_VAL;

SEL_COEFS_SIG : if COEFS_ARE_INPUT_SIGNALS  generate
   COEFS_SRC <= COEFS_IN;
end generate;   

SEL_COEFS_PKG : if COEFS_ARE_PKG_CONSTANT  generate
   ASSIGN_HEXA : if PKG_COEFS_VALUES_ARE_HEXADECIMAL  generate
      LOOPING : for I in 0 to N_TAPS-1  generate
         COEFS_SRC(I) <= COEFS_PKG_HEXA_CONSTANT(I)(COEF_WIDTH-1 downto 0);
      end generate;
   end generate;
   ASSIGN_DECIMAL : if not(PKG_COEFS_VALUES_ARE_HEXADECIMAL)  generate
      LOOPING : for I in 0 to N_TAPS-1  generate
         COEFS_SRC(I) <= conv_std_logic_vector(COEFS_PKG_INTEGER_CONSTANT(I), COEF_WIDTH);
      end generate;
   end generate;
end generate;   

---- Sign extending COEFS
--GEN_EXT_COEFS : if COEF_WIDTH < 18  generate
--   GEN_EXT_COEFS_ARRAY : for I in 0 to N_TAPS-1  generate
--      GEN_EACH_COEF_BIT : for J in 17 downto COEF_WIDTH  generate
--         COEFS_SRC_EXT(I)(J) <= COEFS_SRC(I)(COEF_WIDTH-1);
--      end generate;
--   end generate;   
--end generate;
---- COEFS_SRC_EXT assignment
--GEN_TAP : for I in 0 to N_TAPS-1 generate
--   GEN_BIT : for J in 0 to COEF_WIDTH-1  generate
--      COEFS_SRC_EXT(I)(J) <= COEFS_SRC(I)(J);
--   end generate;
--end generate;

-- Sign extending DIN
GEN_EXT_DIN : if DATA_WIDTH < 18  generate
   GEN_EXT_DIN_BITS : for I in 17 downto DATA_WIDTH  generate
      DIN_EXT(I) <= DIN(DATA_WIDTH-1);
   end generate;
end generate;   
DIN_EXT(DATA_WIDTH-1 downto 0) <= DIN;

CELL_0 : NX_SYSTOLIC_CELL
generic map (
   COEF_WIDTH => COEF_WIDTH,
   DATA_WIDTH => DATA_WIDTH,
   Position   => "First",
   ENABLE_SATURATION_CST => '0',
   SATURATION_RANK_CST => conv_std_logic_vector(55, 6),
   MUX_B_CST  => '0',
   PR_B_CASCADE_MUX_CST => "01",
   MUX_X_CST  => "00"
   )
port map (
   CLK        => CLK,
   SRST       => SRST,
   ENA        => ENA,
   CASC_IN_B  => DIN_EXT,   
   CASC_IN_Z  => ROUND_VAL_EXT,
   COEF       => COEFS_SRC(0), -- COEF_EXTended inside NX_SYSTOLIC_CELL
   OVF        => open,
   CASC_OUT_B  => CASC_B(0),
   CASC_OUT_Z  => CASC_Z(0)
   );

GEN_CELL_1_TO_N_M2 : for I in 1 to N_TAPS-2  generate
   CELL_X : NX_SYSTOLIC_CELL
   generic map (
   COEF_WIDTH => COEF_WIDTH,
   DATA_WIDTH => DATA_WIDTH,
   Position => "",
   ENABLE_SATURATION_CST => '0',
   SATURATION_RANK_CST => conv_std_logic_vector(55, 6),
   MUX_B_CST  => '1',
   PR_B_CASCADE_MUX_CST => "10",
   MUX_X_CST  => "01"
      )
   port map (
      CLK        => CLK,
      SRST       => SRST,
      ENA        => ENA,
      CASC_IN_B  => CASC_B(I-1),
      CASC_IN_Z  => CASC_Z(I-1),
      COEF       => COEFS_SRC(I), -- COEF_EXTended inside NX_SYSTOLIC_CELL
      OVF        => open,
      CASC_OUT_B => CASC_B(I),
      CASC_OUT_Z => CASC_Z(I)
      );
end generate;

CELL_N_M1 : NX_SYSTOLIC_CELL
generic map (
   COEF_WIDTH => COEF_WIDTH,
   DATA_WIDTH => DATA_WIDTH,
   Position => "Last",
   ENABLE_SATURATION_CST => '1',
   SATURATION_RANK_CST => SATURATION_RANK_CST(5 downto 0), --"011101",
   MUX_B_CST  => '1',
   PR_B_CASCADE_MUX_CST => "10",
   MUX_X_CST  => "01"
   )
port map (
   CLK        => CLK,
   SRST       => SRST,
   ENA        => ENA,
   CASC_IN_B  => CASC_B(N_TAPS-2),
   CASC_IN_Z  => CASC_Z(N_TAPS-2),   
   COEF       => COEFS_SRC(N_TAPS-1), -- COEF_EXTended inside NX_SYSTOLIC_CELL
   OVF        => OVF,
   CASC_OUT_B => CASC_B(N_TAPS-1),
   CASC_OUT_Z => CASC_Z(N_TAPS-1)
   );

FIR_OUT <= CASC_Z(N_TAPS-1);

end Behavioral;

