---------------------------------------------------------------------------------------------------------------
-- This design is a parallel FIR filter generator, using the SYSTOLIC structure
--
-- Any N-tap filter (N values between 3 and 56) is impleñented by using a chain of N x DSP blocks
--
-- The filter is fully pipelined. It can reach very high sample frequencies ( > 200 MHz)
--
-- The number of the coefficients (and/or taps) and their respective values must be specified
-- in the "NX_SYSTOLIC_FIR_pack" package, which is part of the "NX_SYSTOLIC_FIR.vhd" file
--
-- Note : once synthesized with 'ExportAsIPCore': 'src/SYSTOLIC_IP_Core.vhd', it can be instantiated 
-- into a global design, just like any other IP Core (see the Python script : 'script_TOP_NX_SYSTOLIC_FIR.py'
---------------------------------------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

library work;
use work.NX_SYSTOLIC_FIR_pack.all;

entity TOP_NX_SYSTOLIC_FIR is
Port ( 
   CLK : in  STD_LOGIC;
   SRST : in  STD_LOGIC;
   ENA : in  STD_LOGIC;
   DIN : in  STD_LOGIC_VECTOR (15 downto 0);
   OVF : out std_logic;
   DOUT : out  STD_LOGIC_VECTOR (15 downto 0)
   );
end TOP_NX_SYSTOLIC_FIR;

architecture Behavioral of TOP_NX_SYSTOLIC_FIR is

component NX_SYSTOLIC_FIR is
generic (
   DATA_WIDTH : integer range 8 to 24;
   COEF_WIDTH : integer range 8 to 24;
   N_TAPS : integer := N_TAPS
   );
Port ( 
   CLK : in  STD_LOGIC;
   SRST : in  STD_LOGIC;
   ENA : in  STD_LOGIC;
   DIN : in  STD_LOGIC_VECTOR (DATA_WIDTH-1 downto 0);
   COEFS_IN : in  COEFS_TYPE;
   OVF : out std_logic;
   FIR_OUT : out  STD_LOGIC_VECTOR (55 downto 0)
   );
end component;

signal COEFS_IN : COEFS_TYPE;
signal DIN_R1, DIN_R2 : std_logic_vector(DATA_WIDTH-1 downto 0);
signal FIR_OUT : STD_LOGIC_VECTOR (55 downto 0);
signal DOUT_INT : STD_LOGIC_VECTOR (15 downto 0);
signal OVF_INT, OVF_INT_R : std_logic;

begin

process(CLK)  begin
   if rising_edge(CLK)  then
      DIN_R1 <= DIN;  -- Input register
      DIN_R2 <= DIN_R1;
   end if;
end process;   

COEFS_IN <= (others => (others => '0'));

FIR : NX_SYSTOLIC_FIR 
generic map (
   DATA_WIDTH => DATA_WIDTH,
   COEF_WIDTH => COEF_WIDTH,
   N_TAPS => N_TAPS
   )
port map ( 
   CLK      => CLK,
   SRST     => SRST,
   ENA      => ENA,
   DIN      => DIN_R2,
   COEFS_IN => COEFS_IN,
   OVF      => OVF_INT,
   FIR_OUT  => FIR_OUT
   );

process(CLK)  begin
   if rising_edge(CLK)  then
      DOUT_int <= FIR_OUT(29 downto 14);  -- truncation 
      DOUT <= DOUT_INT;                   -- Output register 
      OVF_INT_R <= OVF_INT;               -- Registers on OVF to compensate reg delay on DOUT
      OVF <= OVF_INT_R;
   end if;
end process;   

end Behavioral;

