-- =================================================================================================
--   IP_SPW_BANK definition                                                             2018/07/03
-- =================================================================================================

library IEEE;
use IEEE.NUMERIC_STD.all;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_arith.ALL;

library NX;
use NX.nxPackage.all;

entity IP_SPW_BANK is
generic (
    dataSize  : integer range 4 to 10 := 10;     -- data size can be 4, 6, 8 or 10 bits
    bank      : string  := "IOB12";              -- IO bank name
    voltage   : string  := "2.5V";               -- IO bank voltage
    reference : string  := "floating"            -- termination reference, can be floating or VT
);
port (
    DO1    : out std_logic;                                         -- SPW1 D out
    SO1    : out std_logic;                                         -- SPW1 S out
    DI1    : in  std_logic := '0';                                  -- SPW1 D in
    SI1    : in  std_logic := '0';                                  -- SPW1 S in

    RXRST1 : in  std_logic := '0';                                  -- SPW1 RX reset
    RXO1   : out std_logic_vector(19 downto 0);                     -- SPW1 RX data out (2x10 bits)
    RXSCK1 : out std_logic;                                         -- SPW1 RX slow clock

    TXRST1 : in  std_logic := '0';                                  -- SPW1 TX reset
    TXI1   : in  std_logic_vector(9 downto 0) := (others => '0');   -- SPW1 TX data in (10 bits)
    TXFCK1 : in  std_logic := '0';                                  -- SPW1 TX fast clock
    TXSCK1 : out std_logic;                                         -- SPW1 TX slow clock

    DO2    : out std_logic;                                         -- SPW2 D out
    SO2    : out std_logic;                                         -- SPW2 S out
    DI2    : in  std_logic := '0';                                  -- SPW2 D in
    SI2    : in  std_logic := '0';                                  -- SPW2 S in

    RXRST2 : in  std_logic := '0';                                  -- SPW2 RX reset
    RXO2   : out std_logic_vector(19 downto 0);                     -- SPW2 RX data out (2x10 bits)
    RXSCK2 : out std_logic;                                         -- SPW2 RX slow clock

    TXRST2 : in  std_logic := '0';                                  -- SPW2 TX reset
    TXI2   : in  std_logic_vector(9 downto 0) := (others => '0');   -- SPW2 TX data in (10 bits)
    TXFCK2 : in  std_logic := '0';                                  -- SPW2 TX fast clock
    TXSCK2 : out std_logic                                          -- SPW2 TX slow clock
);
end IP_SPW_BANK;


`protect begin_protected
`protect version = 1
`protect encrypt_agent = "NanoXplore"
`protect key_keyowner = "NanoXplore", key_keyname = "NX-IP-RSA-1", key_method = "rsa"
`protect encoding = (enctype = "base64", line_length = 64, bytes = 128)
`protect key_block
l4og7qTueZGzo9B6p6SIFrJysYcn1E0BcqlOgglLdYPX9hji4VfQxvEgVGbpVDQX
t9IBhDfU5umLHGU4YSMn79BnedLF4HmGKCmknDD0HNrjCdMIkHHXVnp4rz6CEa0c
o1EFBd/8i5VXaszX1P2MGrsPGbHDq5AKICM0walp4YY=
`protect key_keyowner = "Mentor Graphics Corporation", key_keyname = "MGC-VERIF-SIM-RSA-1", key_method = "rsa"
`protect encoding = (enctype = "base64", line_length = 64, bytes = 128)
`protect key_block
Nx2A01CA3EF52w3iTRPn7oLp2w1J/S5TuGpehFD4nCrlC5dpSTZ8zt9Ae5iPevlL
r6uIw/XouQKYFobfWvRA+l+L+jcixDoOoKMF+5hBWjtfD+SOSW4bi3Hql64OxL+L
/eZbxLJV8fc/zdHoqnIsxVBtan6+n4Y+NUooO+1spY0=
`protect data_method = "aes128-cbc"
`protect encoding = (enctype = "base64", line_length = 64, bytes = 19328)
`protect data_block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`protect end_protected

